%% samplecompiler
% 
% Author: Brian C. Wise
% Version: July 2023
% 
% Application: quantification of glycosaminoglycan (GAG) staining at the
% Achilles tendon insertion
% 
% Purpose: this code calculates and compiles average H,S,V data within each 
% ROI across all serial tissue sections for an entire sample using the 
% H,S,V data provided by HSV.m.
% 
% Inputs
% 
% filename: character string corresponding to the filename of a 24-bit RGB 
% color image contained in the Current Folder
% 
% varargin: a variable number (z) of 11 x 4 cell arrays output from the 
% HSV.m function, one cell array for each section within a sample. 
% First row and column are headers. 
% Average H,S,V data within rows 2:11, columns 2:4.
% 
% NOTE: provide cell arrays in order of increasing section level (tissue
% depth).
% 
% Outputs 
% 
% [a,b]
% 
% 'a' provides an 11 x 4 cell array with first row and column containing 
% headers, and H,S,V sample averages provided in rows 2:11, columns 2:4.
% 
% 'b' provides a 11 x 4 x z page of cell arrays. Each 11 x 4 cell array 
% provided as input is stacked along the z dimension. 
% 
% This function automatically saves 'a' as a .xlsx file

function [a,b] = samplecompiler(samplename,varargin)

% Preallocate a 10 x 3 x z matrix to place the data within the cell arrays 
% provided as input. 
HSVstack = zeros(10,3,nargin-1);

% Preallocate cell arrays to tabulate row and column headers
colheaders = cell(1,4,nargin-1);
rowheaders = cell(10,1,nargin-1);

% Go through cell arrays provided as input and pull out HSV data.
% Place in HSVstack.
for i = 1:(nargin-1)
    HSVstack(:,:,i) = cell2mat(varargin{i}(2:11,2:4));
    colheaders(:,:,i) = varargin{i}(1,:);
    rowheaders(:,:,i) = varargin{i}(2:11,1);
end

% Preallocate a cell array to tabulate sample averages of H,S,V data
samplecentroids = cell(11,4);

% Transfer the column headers from the cell arrays provided as input
samplecentroids(1,2:4) = varargin{1}(1,2:4);

% Transfer the row headers from the cell arrays provided as input
samplecentroids(2:11,1) = varargin{1}(2:11,1);

% Add sample name to row 1, column 1
samplecentroids(1,1) = {samplename};

% Calculate sample averages along z dimension of HSVstack and place in
% this dummy cell array
samplecentroids(2:11,2:4) = num2cell(mean(HSVstack,3));

% Save sample average H,S,V data as: 'samplename_HSV.xlsx'
fileout = sprintf('%s%s.xlsx',samplename,'_HSV');
writecell(samplecentroids,fileout);
a = samplecentroids;

% Preallocate a cell array to compile the H,S,V data provided as input
compileddata = cell(11,4,nargin-1);

% Transfer the column headers from the cell arrays provided as input
compileddata(1,:,:) = colheaders;

% Transfer the row headers from the cell arrays provided as input
compileddata(2:11,1,:) = rowheaders;

% Place H,S,V data stacked along z into dummy cell array
compileddata(2:11,2:4,:) = num2cell(HSVstack);
b = compileddata;

end