%% pairwise
%
% Author: Brian C. Wise
% Version: July 2023
%
% Application: quantification of glycosaminoglycan (GAG) staining at the
% Achilles tendon insertion
%
% Purpose: this code performs data normalization with respect to the 
% periosteal fibrocartilage using sample average H,S,V data provided by
% samplecompiler.m.
% 
% Inputs
% 
% sample1name: character string for the name of sample 1
%
% sample1data: 11 x 4 x z cell array provided by samplecompiler.m. Each
% 11 x 4 cell array stacked along the 3rd dimension (z) contains H,S,V data
% for individual tissue sections in sample 1. 
% 
% sample2name: character string for the name of sample 2
% 
% sample2data: 11 x 4 x z cell array provided by samplecompiler.m. Each
% 11 x 4 cell array stacked along the 3rd dimension (z) contains H,S,V data
% for individual tissue sections in sample 2.
%
% NOTE: the value of z (i.e. # of serial tissue sections) should be
% equivalent for both samples. Additionally, data should be organized by
% increasing section level (tissue depth). Sections from each pair should
% be chosen at matched levels for staining and analysis. In this way, data
% along z are level-matched between sample pairs. 
%
% Output
%
% 11 x 11 cell array. First row and column contain header information.
% Columns 2:4 contains average H,S,V data for sample 1 normalized to
% average H,S,V for the periosteal fibrocartilage (PF). Columns 5:7
% contains average H,S,V data for sample 2 normalized to average H,S,V for
% the periosteal fibrocartilage. Columns 8 and 9 contain Euclidean distance
% data comparing sample 1 and sample 2 to the PF, respectively. Columns 10
% and 11 contain fibrocartilage scores for sample 1 and sample 2,
% respectively.

function out = pairwise(sample1name,sample1data,sample2name,sample2data)

% Extract hue data for all ROIs in sample 1.
sample1H = cell2mat(sample1data(2:11,2,:));

% Convert sample 1 hue data to radians.
sample1Hrad = deg2rad(sample1H);

% Extract saturation data for all ROIs in sample 1.
sample1S = cell2mat(sample1data(2:11,3,:));
 
% Extract value data for all ROIs in sample 1.
sample1V = cell2mat(sample1data(2:11,4,:));

% Extract hue data for all ROIs in sample 2.
sample2H = cell2mat(sample2data(2:11,2,:));

% Convert sample 2 hue data to radians.
sample2Hrad = deg2rad(sample2H);

% Extract saturation data for all ROIs in sample 2.
sample2S = cell2mat(sample2data(2:11,3,:));

% Extract value data for all ROIs in sample 2.
sample2V = cell2mat(sample2data(2:11,4));

% Preallocate a cell array to tabulate our normalized data.
euclideannorms = cell(11,11);

% Average the hue, saturation and value data in the periosteal
% fibrocartilage across samples for data normalization.
% Preallocate a matrix to compile periosteal fibrocartilage (PF) data
% for both samples such that row 1 = sample 1 and row 2 = sample 2, columns
% contain H,S,V data, and data for tissue sections are stacked along z.
sz = size(sample1H); % This will provide the number of tissue sections.
PFdata = zeros(2,3,sz(3));

% Tabulate the matrix with H,S,V data for the periosteal fibrocartilage.
PFdata(1,1,:) = sample1H(10,1,:); PFdata(1,2,:) = sample1S(10,1,:); PFdata(1,3,:) = sample1V(10,1,:);
PFdata(2,1,:) = sample2H(10,1,:); PFdata(2,2,:) = sample2S(10,1,:); PFdata(2,3,:) = sample2V(10,1,:);

% Calculate average PF H,S,V between samples for each tissue section.
avgPFdata = mean(PFdata,1); % 1 x 3 x z
% Replicate this data into a 10 x 3 x z matrix that will be used to
% normalize sample data.
PFarray = repmat(avgPFdata,10,1); % Avg PF H,S,V data repeated in each row

% Extract H,S,V data for each sample.
sample1HSV = cell2mat(sample1data(2:11,2:4,:));
sample2HSV = cell2mat(sample2data(2:11,2:4,:));

% Divide H,S,V data for each sample by average PF H,S,V. Place normalized
% sample 1 data in columns 2:4 and normalized sample 2 data in columns 5:7.
euclideannorms(2:11,2:4) = num2cell(mean(sample1HSV./PFarray,3));
euclideannorms(2:11,5:7) = num2cell(mean(sample2HSV./PFarray,3));

% Within each ROI, the average hue and saturation represent coordinate
% positions and the 2-dimensional hue-saturation color space. Therefore,
% the Euclidean distance separating the average hue and saturation between
% two ROIs of interest can be calculated. This describes the distance
% separating the average color between the ROIs being compared within the
% hue-saturation color space. 
% 
% The Euclidean distance separating the average hue and saturation in each 
% ROI and the average hue and saturation in the periosteal fibrocartilage 
% provides a normalized measure describing how the stain color in each ROI 
% compares to that of the PF. If this distance decreases, the stain color
% in the ROI is approaching that of the PF suggesting increased GAG
% staining and fibrocartilage formation. This Euclidean distance data
% normalizing each ROI to the PF can be used to calculate a fibrocartilage 
% score as described in the protocol.
%
% Fibrocartilage Score = (255 - euclidean distance)/255
% 
% 255 is the radius of the 2-D hue-saturation color space.
% 
% The equation for calculating Euclidean distance is provided in the 
% protocol.
%
% Calculate Euclidean distances separating average hue and saturation data
% in each ROI of sample 1 to those of the PF, then average across all
% tissue sections (stacked in z) to calculate sample averaged Euclidean
% distance data for each ROI. Convert to a cell array and place the data
% into column 8 of the preallocated matrix being used to compile data.
avgPFHrad = deg2rad(avgPFdata(1,1,:)); % Convert PF hue data to radians
euclideannorms(2:11,8) = num2cell(mean((sample1S.^2 + avgPFdata(1,2,:).^2 - ...
    2*sample1S.*avgPFdata(1,2,:).*cos(sample1Hrad-avgPFHrad)).^0.5,3));

% Calculate Euclidean distances separating average hue and saturation data
% in each ROI of sample 2 to those of the PF, then average across all
% tissue sections (stacked in z) to calculate sample averaged Euclidean
% distance data for each ROI. Convert to a cell array and place the data
% into column 9 of the preallocated matrix being used to compile data.
euclideannorms(2:11,9) = num2cell(mean((sample2S.^2 + avgPFdata(1,2,:).^2 - ...
    2*sample2S.*avgPFdata(1,2,:).*cos(sample2Hrad-avgPFHrad)).^0.5,3));

% Now calculate fibrocartilage scores as described above.
euclideannorms(2:11,10) = num2cell((255-cell2mat(euclideannorms(2:11,8)))/255);
euclideannorms(2:11,11) = num2cell((255-cell2mat(euclideannorms(2:11,9)))/255);

% Add row headers 
euclideannorms(2:11,1) = {'Total CTF','Quadrant 1','Quadrant 2',...
    'Quadrant 3','Quadrant 4','Distal','Proximal','Deep','Superficial',...
    'Periosteal Fibrocartilage'};

% Add column headers
euclideannorms(1,2:11) = {'Sample1 H (normalized)','Sample1 S (normalized)',...
    'Sample1 V (normalized)','Sample2 H (normalized)','Sample2 S (normalized)',...
    'Sample2 V (normalized)','Avg Euclidean Distance Sample1 and PF',...
    'Avg Euclidean Distance Sample2 and PF','Sample1 Fibrocartilage Score'...
    'Sample2 Fibrocartilage Score'};

out = euclideannorms;

% Save normalized data as 'sample1name_sample2name_comparison.xlsx'.
fileout = sprintf('%s%s%s%s.xlsx',sample1name,'_',sample2name,'_comparison');
writecell(euclideannorms,fileout);

end