%% ImportROIs
%
% Author: Brian C. Wise
% Version: July 2023
% 
% Application: quantification of glycosaminoglycan (GAG) staining at the
% Achilles tendon insertion
%
% Purpose: this function can be used to import previously defined ROI
% boundaries saved as .xlsx files using CTF_outliner.m into the current
% MATLAB workspace
%
% Input
% 
% filename: character string describing the filename of a .xlsx file within
% the Current Folder containing ROI boundary coordinates defined using
% CTF_outliner.m.
%
% Output
% 
% 1 x 10 cell array containing ROI coordinates:
% {totalCTF,quad1,quad2,quad3,quad4,distal,proximal,deep,superficial,PF}

function out = importROIs(filename)

% Use readtable() to read in the .xlsx file as a table.
fulldata = readtable(filename,'readvariablenames',false);

% Convert table to cell array.
datacell = table2cell(fulldata);
datacell(cellfun(@ischar,datacell))={};

% Pull ROI coordinates out of cell array as matrices.
TOTAL = cell2mat(datacell(:,1:2));
QUAD1 = cell2mat(datacell(:,3:4));
QUAD2 = cell2mat(datacell(:,5:6));
QUAD3 = cell2mat(datacell(:,7:8));
QUAD4 = cell2mat(datacell(:,9:10));
DISTAL = cell2mat(datacell(:,11:12));
PROXIMAL = cell2mat(datacell(:,13:14));
DEEP = cell2mat(datacell(:,15:16));
SUPERFICIAL = cell2mat(datacell(:,17:18));
PF = cell2mat(datacell(:,19:20));

% Remove missing entries.
total = rmmissing(TOTAL);
quad1 = rmmissing(QUAD1);
quad2 = rmmissing(QUAD2);
quad3 = rmmissing(QUAD3);
quad4 = rmmissing(QUAD4);
distal = rmmissing(DISTAL);
proximal = rmmissing(PROXIMAL);
deep = rmmissing(DEEP);
superficial = rmmissing(SUPERFICIAL);
pf = rmmissing(PF);

% Compile ROI coordinate matrices into a 1 x 10 cell array and output to
% the MATLAB workspace.
out = {total quad1 quad2 quad3 quad4 distal proximal deep superficial pf};

end