%% HSV
% 
% Author: Brian C. Wise
% Version: July 2023
% 
% Application: quantification of glycosaminoglycan (GAG) staining at the
% Achilles tendon insertion
% 
% Purpose: this code performs color image analysis within each ROI as
% defined by CTF_outliner.m by converting RGB pixel intensity data to the
% HSV color space to quantify average hue, saturation and value within each
% ROI.
% 
% Inputs 
% 
% filename: character string corresponding to the filename of a 24-bit RGB 
% color image contained in the Current Folder
% 
% ROIs: 1 x 10 cell array output by CTF_outliner.m containing ROI coordinates: 
% {totalCTF,quad1,quad2,quad3,quad4,distal,proximal,deep,superficial,PF}
% 
% Output
% 
% 11 x 4 cell array, with headers in the first row and column. Average 
% H,S,V data for each ROI is contained in rows 2:11, columns 2:4.

function out = HSV(filename,ROIs)

% Read in the image. For a 24-bit RGB color image that is X x Y pixels in 
% size, imread() will create a X x Y x 3 array of uint8 integers. Red pixel
% intensities are contained in (X,Y,1), blue in (X,Y,2), and green in
% (X,Y,3). Pixel intensities are 8-bit ranging from 0 to 255 (2^8 - 1).
RGB = imread(filename);

% Separate the red, green and blue pixel intensity data.
R = double(RGB(:,:,1));
G = double(RGB(:,:,2));
B = double(RGB(:,:,3));

% Calculate the maximum and minimum intensities for each pixel.
maxRGB = max(B,max(R,G));
minRGB = min(B,min(R,G));

% Convert the R,G,B intensities for each pixel into hue, saturation, and 
% value (HSV color space).
% First, preallocate a matrix to compile hue data
H = zeros(size(R));

% R,G,B values are defined on a linear scale from 0 to 255. Hues in HSV are 
% defined around a 0-360 degree circle.
% Conditional statements for R,G,B determine the transformation equation: 
% If R is max, G>=B: use Eqn 1 (see below)
% If R is max, G<B: use Eqn 2 (see below)
% If G is max: use Eqn 3 (see below)
% If B is max: use Eqn 4 (see below)

% Define a set of logical matrices for each conditional statement. 
% For example, for 'R is max', generate a matrix with values of 1 assigned
% to pixels in which maximum RGB intensity = R, and 0 elsewhere. For 
% G >= B, generate a matrix with values of 1 assigned to pixels in which 
% green intensity >= blue intensity, and 0 elsewhere. Etc. By multiplying
% these logical matrices together, the combinatorial conditional statements
% above can be implemented.

% Compare the matrix of maximum pixel intensities to the matrix of red
% pixel intensities and generate a logical matrix in which any pixel
% satisfying max(R,G,B) = max(R) is assigned a value of 1 and all other
% pixels are assigned a value of 0.
Rmax = maxRGB == R;

% Using a similar approach, create a logical matrix in which any pixel 
% satisfying green intensity >= blue intensity is assigned a value of 1 and
% all other pixels are assigned a value of 0.
Gb = G >= B;

% Create another logical matrix in which any pixel satisfying
% green intensity < blue intensity is assigned a value of 1 and all other 
% pixels are assigned a value of 0.
Bg = G < B;

% Create another logical matrix in which any pixel satisfying 
% max(R,G,B) = max(G) is assigned a value of 1 and all other pixels are
% assigned a value of 0.
Gmax = maxRGB == G;

% Create another logical matrix in which any pixel satisfying 
% max(R,G,B) = max(B) is assigned a value of 1 and all other pixels are
% assigned a value of 0.
Bmax = maxRGB == B;

% It is possible for R=G=B. Following the conditional statements, all 3
% equations for transforming to HSV provide the same result. To avoid
% repeating calculations and adding them together in the steps that follow,
% find locations where R=G=B and arbitrarily keep Rmax = 1, but change
% Bmax = 0 and Gmax = 0. Otherwise, Rmax = Gmax = Bmax = 1. 
RGBmax = Rmax+Gmax+Bmax == 3;
Gmax = Gmax-RGBmax;
Bmax = Bmax-RGBmax;

% Similarly, if red intensity is not greatest, it is possible that G=B and
% both transformation equations indicated by the conditional statements 
% provide the same result. Find locations where Gmax = 1 and Bmax = 1, and 
% set Bmax = 0.
GBmax = Gmax+Bmax == 2;
Bmax = Bmax-GBmax;

% Similarly, maximum intensity could correspond to red and blue (R=B=max).
% To avoid applying two equations, find where Rmax = Bmax and set Bmax = 0.
RBmax = Rmax+Bmax == 2;
Bmax = Bmax-RBmax;

% Now, use the logical matrices to calculate hues according to the
% conditional equations. In this approach, for example, hue is calculated
% using conditional Eqn 1 (R = max, G >= B) for EVERY pixel regardless of
% whether that condition is met. Then, multiply with the logical matrix
% Rmax and the logical matrix Gb (G>=b). For a pixel, if these conditions 
% are not met, the hue value is multiplied by zero. This will yield a 
% matrix of calculated hues for pixels where R=max,G>=B, and zeroes
% everywhere else. Repeat that for each condition and add all the results
% together to obtain a matrix of hue values.

% Calculate hues according to the 4 conditional statements below.
c1 = ((G-B)./(maxRGB-minRGB))*60; %Eqn 1
H = H + c1.*Rmax.*Gb;

c2 = (((G-B)./(maxRGB-minRGB))*60)+360; %Eqn 2
H = H + c2.*Rmax.*Bg;

c3 = (((B-R)./(maxRGB-minRGB))*60)+120; %Eqn 3
H = H + c3.*Gmax;

c4 = (((R-G)./(maxRGB-minRGB))*60)+240; %Eqn 4
H = H + c4.*Bmax;

% Calculate saturation data.
S = ((maxRGB-minRGB)./maxRGB)*255;

% Calculate value data.
V = maxRGB;

% Preallocate a cell array to compile average H,S,V data for each ROI. 
centroids = cell(11,4);
% Add in row and column headers
centroids(1,2:4) = {'H','S','V'};
centroids(:,1) = {filename,'Total CTF','Quadrant 1','Quadrant 2',...
    'Quadrant 3','Quadrant 4','Distal','Proximal','Deep','Superficial',...
    'Periosteal Fibrocartilage'};

% Calculate averages and tabulate this cell array using a for loop
% to cycle through each ROI, placing data in rows 2:11, columns 2:4.
% 
% For each ROI, averages are calculated by using ROI coordinates to create
% a masked image in which all values outside the ROI = NaN. H,S,V data can
% then be averaged across the entire image using the mean() function and
% specifying 'omitnan' to calculate averages specifically within each ROI. 

for i = 1:length(ROIs)
    imshow(RGB);
    h = images.roi.Polygon(gca,'Position',ROIs{1,i});
    maskedImage = createMask(h);

    croppedH = H;
    croppedH(~maskedImage) = NaN;
    croppedS = S;
    croppedS(~maskedImage) = NaN;
    croppedV = V;
    croppedV(~maskedImage) = NaN;
    Hmean = mean(croppedH,'all','omitnan');
    Smean = mean(croppedS,'all','omitnan');
    Vmean = mean(croppedV,'all','omitnan');

    centroids(i+1,2:4) = num2cell([Hmean, Smean, Vmean]);
    close all
end

% Output the cell array containing average H,S,V data for each ROI into the
% MATLAB workspace. 
out = centroids;

% Obtain the base filename using fileparts().
[folder, baseFileNameNoExt, extension] = fileparts(filename);
% Generate a new filename: 'filename_HSV.xlsx'
fileout = sprintf('%s%s.xlsx',baseFileNameNoExt,'_HSV');
% Save average H,S,V data as .xlsx file
writecell(centroids,fileout);

end