%% CTF Outliner
% 
% Author: Brian C. Wise
% Version: July 2023 
%
% Application: quantification of glycosaminoglycan (GAG) staining at the
% Achilles tendon insertion
% 
% Purpose: this code assists the user in manually defining a region of 
% interest (ROI) encompassing the compressive tendon fibrocartilage (CTF) 
% at the Achilles tendon insertion, which is further subdivided into 4 
% quadrants for spatial analysis. This generates 10 ROIs:
% 
% Total CTF: ROI capturing entire CTF
% Quadrant 1: distal deep
% Quadrant 2: proximal deep
% Quadrant 3: distal superficial
% Quadrant 4: proximal superficial
% Distal: quadrant 1 & quadrant 3 combined
% Proximal: quadrant 2 & quadrant 4 combined
% Deep: quadrant 1 & quadrant 2 combined
% Superficial: quadrant 3 & quadrant 4 combined
% 
% Additonally, the code will allow a small ROI to be defined within the 
% periosteal fibrocartilage (PF) lining the calcaneus, which provides a 
% ROI for internal normalization of GAG staining
%
% Input
% 
% filename: character string corresponding to the filename of a 24-bit RGB 
% color image contained in the Current Folder
% 
% Output
% 
% 1 x 10 cell array containing ROI coordinates:
% {totalCTF,quad1,quad2,quad3,quad4,distal,proximal,deep,superficial,PF}
% 
% In each cell, n pairs of (x,y) coordinates defining a ROI are compiled
% into n x 2 matrices. 
%
% The ROI coordinates are automatically saved as a .xlsx into the Current
% Folder as 'filename_boundaries.xlsx'.
% 
% During execution, the original RGB image will be displayed with the 
% user-defined ROIs overlaid and outlined in red. This image with ROIs
% overlaid will be automatically saved as a .tif into the Current Folder
% as 'filename_boundaries.tif'. 

function out = CTF_outliner(filename)

% Read in the image. For a 24-bit RGB color image that is X x Y pixels in 
% size, imread() will create a X x Y x 3 array of uint8 integers. Red pixel
% intensities are contained in (X,Y,1), blue in (X,Y,2), and green in
% (X,Y,3). Pixel intensities are 8-bit ranging from 0 to 255 (2^8 - 1).
RGB = imread(filename);

% To define ROIs, this code utilizes several features of MATLAB's Image 
% Processing Toolbox that allow the user to interactively draw ROIs
% directly over a displayed image.
% 
% To define a ROI outlining the total CTF, first define the deep 
% boundary by drawing a series of connected line segment using 
% drawpolyline() along the deep tendon border, beginning distally at
% the junction of the Achilles tendon and calcaneus and extending 
% proximally 800 microns.

figure(1)
imshow(RGB); % Display the image

% Using drawpolyline(), each mouse click on the displayed image selects a
% pixel, and each successive selection is connected to the previous by a 
% line segment that is overlaid on the image in real-time (below in color 
% red). The function is terminated by a double mouse click. The resulting 
% polyline object is defined by a collection of pixel coordinates. 
% 
% Begin by clicking the point where the deep Achilles tendon intersects 
% with the calcaneus. Moving proximally, continue to click points to draw 
% a series of line segments that approximates the curvature of the deep 
% tendon border. Continue to select points until a polyline has been drawn 
% that extends well into the tendon midsubstance towards the myotendinous 
% junction. When finished, double click.

deep = drawpolyline('Color','red'); 

% Extract the pixel coordinates defining the polyline. For a polyline
% defined by k points, this will provide a k x 2 matrix of (x,y)
% coordinates in each row.
deepverts = deep.Position; 

% Now, trim down the polyline such that it extends only 800 microns 
% proximal from the Achilles-calcaneus junction. To do this, first 
% calculate the length of the line segments connecting each pair of
% pixel coordinates defining in the polyline: 

% segment length = sqrt((delta_x)^2 + (delta_y)^2)

% Calculate delta_x and delta_y between adjacent points. 
deepdeltas = deepverts(2:end,:)-deepverts(1:end-1,:); 
% Use this to calculate segment lengths.
deepseglengths = sum(deepdeltas.^2,2).^0.5;

% In this (k-1) x 1 matrix, row 1 = segment length between point 1
% and 2, row 2 = segment length between point 2 and 3, etc. Use cumsum()
% to calculate the cumulative length of the polyline. This provides a new
% matrix where row 1 = segment length between point 1 and 2, row 2 =
% segment length between point 1-3, row 3 = segment length between point
% 1-4, etc. 
deeplength = cumsum(deepseglengths);

% Now, identify the location where the cumulative segment length exceeds 
% 800 microns. For the sample image provided, this corresponds to 727.27 
% pixels. 
deepfinder = deeplength - 727.27;

% In this matrix, the transition from negative to positive values indicates
% the cumulative segment length has surpassed 800 microns. Create a logical 
% matrix where negative values = zero and positive values = 1. 
deepfinderindex = deepfinder < 0 == 0;

% Find the location of the first 1, which represents the line segment in
% the polyline that contains a point 800 microns distal to the 
% Achilles-calcaneal junction.
deepproxloc = find(deepfinderindex,1);

% This provides the two points in the polyline between which 800 microns in
% cumulative segment length exists, as it likely isn't defined perfectly by
% one of the points defined by the user. Trim back the original polyline to 
% the distal point in this pair, which will fall just shy of 800 microns.
deeptrimmed = deepverts(1:deepproxloc,:);

% Now, interpolate to extend from this point and reach a cumulative segment 
% length of 800 microns. 
% Calculate the length needed to extend from this coordinate position to 
% achieve 800 microns in total length.
deepextensionlength = 727.27-deeplength(deepproxloc-1);

% Calculate the incremental change in x and y that must be added to the 
% trimmed deep boundary to extend to 800 microns in cumulative length.
y800 = (deepextensionlength/deepseglengths(deepproxloc))*(deepverts(deepproxloc+1,2)-deepverts(deepproxloc,2));
x800 = (deepextensionlength/deepseglengths(deepproxloc))*(deepverts(deepproxloc+1,1)-deepverts(deepproxloc,1));

% Calculate the x and y coordinates of the new position along the polyline
% that trims the distal boundary down to 800 microns in length
deepextensionvert = [deepverts(deepproxloc,1)+x800 deepverts(deepproxloc,2)+y800];

% Assign the coordinates that define this deep boundary to a variable
% 'deepboundary'
deepboundary = [deeptrimmed;deepextensionvert];

close all

% Now, define the proximal boundary of the total CTF. This boundary is a
% line segment that connects the proximal point of the deep boundary (800
% microns from the Achilles-calcaneus junction) to the superficial border
% along an orientation approximately perpendicular to the local fiber
% alignment.

% Again display the image, with the deep boundary (defined above) overlaid.
figure(2)
imshow(RGB); hold on
plot(deepboundary(:,1),deepboundary(:,2),'r'); hold off

% At the proximal end of this deep boundary (800 microns from the
% Achilles-calcaneus junction), use the drawline() function to
% draw a single line segment approximately parallel to the local fiber 
% alignmnent. The drawline() function utilizes a click-drag-release 
% technique: click to define the first point and HOLD, drag the cursor to 
% the second point, and release to define the segment.
fiber = drawline('Color','r');

% Extract the pixel coordinates for the line segment defining the local
% fiber orientation.
fiberpos = fiber.Position;

% Calculate slope of the local fiber segment
fiberslope = (fiberpos(2,2)-fiberpos(1,2))/(fiberpos(2,1)-fiberpos(1,1));

close all

% Insert a trace of the proximal boundary by defining a line that passes
% through the proximal point of the deep boundary and runs perpendicular to
% the local fiber alignment using the fiber slope calculated above. 

% Define an inline function 'trace' that uses the point-slope formula for a
% line: y = m(x-x0)+y0. Here, m is the slope perpendicular to the
% fiber alignment (i.e. (-1/fiber slope)), and x0 and y0 are the
% coordinates of the most proximal point of the deep boundary.
trace = @(x) (-1/fiberslope)*(x-deepextensionvert(1,1))+deepextensionvert(1,2);

% The trace of the proximal boundary will follow the above defined equation
% and pass through the RGB image. Determine the size of the image in
% pixels.
imagesize = size(RGB);

% Define the x-range of the RGB image
xtrace = 0:1:imagesize(1);

% Provide the x-range to the point-slope equation in the trace function to
% determine the corresponding y-values.
ytrace = trace(xtrace);

% Together, the coordinates provided by (xtrace,ytrace) define a line 
% perpendicular to the local fiber alignment, but likely extend outside the
% y-range of the RGB image. Trim the xtrace and ytrace values to only
% include values of ytrace within the y-range of the RGB image.
xtracetrimmed = xtrace(ytrace>0 & ytrace<imagesize(2));
ytracetrimmed = ytrace(ytrace>0 & ytrace<imagesize(2));

% Now, overlay this trace of the proximal boundary in a red dashed line
% overtop the original image, along with the deep boundary defined above.
figure(3)
imshow(RGB); hold on
plot(deepboundary(:,1),deepboundary(:,2),'r');
plot(xtracetrimmed,ytracetrimmed,'r--'); hold off

% Use the drawpoint() function to select a point at the intersection of the
% proximal boundary trace (red dashed) and the superficial border of the
% tendon.
superproximal = drawpoint('Color','r');
% Extract the coordinate position of this point. 
superproximalpoint = superproximal.Position;

% This point, together with the most proximal point of the deep boundary,
% define a line segment representing the proximal boundary of the total 
% CTF. Combine both of these coordinates in a matrix.
proximalboundary = [deepextensionvert;superproximalpoint];

close all

% Next, define the distal boundary by a line segment that separates the CTF
% region from the adjacent attachment zone fibrocartilage (AZF) along the
% tidemark, easily identifiable by a distinct color change.

% Display the image and overlay the deep and proximal boundaries. Again 
% using the drawline() function, draw a line segment connecting the 
% distal point of the deep boundary (at the Achilles-calcaneus junction) to
% the superficial tendon border along the tidemark such that the entirety
% of the AZF lies distal to this line segment.
figure(4)
imshow(RGB); hold on
plot(deepboundary(:,1),deepboundary(:,2),'r');
plot(proximalboundary(:,1),proximalboundary(:,2),'r'); hold off
distalline = drawline('Color','r');

% Extract the coordinate positions for this line segment defining the
% distal boundary of the total CTF.
distalboundary = distalline.Position;

close all

% The final step in establishing the total CTF ROI is to define the
% superficial boundary. This is achieved by using the drawpolyline()
% function to trace the superficial tendon border and connect the
% superficial point of the distal boundary to the superficial point of the
% proximal boundary.

% Display the image and overlay the deep, proximal and distal boundaries.
% Starting at the superficial point of the distal boundary, use
% drawpolyline() to draw a series of connect line segments to trace the
% curvature of the superficial tendon border, moving proximally until
% reaching the superficial point of the proximal boundary. Double click to
% finish.
figure(5)
imshow(RGB); hold on
plot(deepboundary(:,1),deepboundary(:,2),'r');
plot(proximalboundary(:,1),proximalboundary(:,2),'r');
plot(distalboundary(:,1),distalboundary(:,2),'r'); hold off
superficialline = drawpolyline('Color','Red');
%Assign the vertices of this superficial boundary to variable
%'superficialverts'
superficialverts = superficialline.Position;

close all

% Compile all the coordinates defnining the total CTF ROI into a matrix.
boundary = [deepboundary(2:end,:);proximalboundary(2:end,:);...
    flip(superficialverts(2:end-1,:));flip(distalboundary);deepboundary(2,:)];

% Now, divide this total CTF ROI into 4 quadrants, guided by the midpoints
% of each of the 4 outer boundaries defined above. For the deep boundary, 
% the midpoint lies at 400 microns consistent with the approach
% outlined above. To identify this location along the polyline, subtract
% 400 microns (363.64 pixels for the sample image provided) from the vector
% of cumulative segment length along the deep boundary. 
deephalffinder = deeplength - 363.64;

% Again, the transition from negative to positive identifies the segment
% containing the deep midpoint. Define a logical array where any negative 
% value = 0 and any positive = 1
deephalffinderindex = deephalffinder < 0 == 0;

% Find the first value = 1, which identifies the segment containing the
% deep midpoint.
deephalfloc = find(deephalffinderindex,1);

% Trim the deep boundary back to the point immediately distal to the 
% midpoint, just shy of 400 microns. This matrix contains the coordinates 
% for the distal half of the deep boundary (without the midpoint). 
distaldeephalftrimmed = deepverts(1:deephalfloc,:);

% The remaining coordinates represent the proximal half of the deep
% boundary (without the midpoint).
proximaldeephalftrimmed = deepboundary(deephalfloc+2:end,:);

% Interpolating, calculate the distance needed to extend to the midpoint
% at 400 microns.
deephalfextensionlength = 363.64-deeplength(deephalfloc-1);

% Calculate the x- and y-distance needed to extend to reach the midpoint.
y400 = (deephalfextensionlength/deepseglengths(deephalfloc))*(deepverts(deephalfloc+1,2)-deepverts(deephalfloc,2));
x400 = (deephalfextensionlength/deepseglengths(deephalfloc))*(deepverts(deephalfloc+1,1)-deepverts(deephalfloc,1));
% This coordinate provides the midpoint at 400 microns along the deep 
% boundary.
deepextensionhalfvert = [deepverts(deephalfloc,1)+x400 deepverts(deephalfloc,2)+y400];

% In a similar fashion, calculate the midpoint of the superficial
% boundary. To do this, first determine the total length of the superficial
% boundary by calculating and adding together the individual segment 
% lengths of the polyline.

% Calculate and sum the segment lengths
superdeltas = superficialverts(2:end,:)-superficialverts(1:end-1,:);
superseglengths = sum(superdeltas.^2,2).^0.5;

% Calculate the cumulative segment length of the superficial boundary from 
% one point to the next moving proximally using cumsum().
superlength = cumsum(superseglengths);

% Calculate half of the total length of the superficial boundary, and 
% subtract this from the cumulative lengths. Once again, the transition
% from negative to positive in this matrix identifies the segment
% containing the midpoint.
superfinder = superlength - superlength(end)/2;

% Define a logical array where any negative value = 0, positive = 1.
superfinderindex = superfinder < 0 == 0;

% Use this to find the segment containing the midpoint of the superficial 
% boundary.
superproxloc = find(superfinderindex,1);

% As before, trim the superficial boundary back to the point immediately 
% distal to the midpoint. These coordinates define the distal half of the
% superficial boundary (without the midpoint).
distalsupertrimmed = superficialverts(1:superproxloc,:);

% The remaining coordinates define the proximal half of the superficial
% boundary (without the midpoint).
proximalsupertrimmed = superficialverts(superproxloc+1:end,:);

% Interpolating, calculate the length needed to extend to the midpoint.
superextensionlength = superlength(end)/2 - superlength(superproxloc-1);

% Calculate the x- and y-distance needed to extend to the midpoint.
superhalfy = (superextensionlength/superseglengths(superproxloc))*(superficialverts(superproxloc+1,2)-superficialverts(superproxloc,2));
superhalfx = (superextensionlength/superseglengths(superproxloc))*(superficialverts(superproxloc+1,1)-superficialverts(superproxloc,1));
% This coordinate represents the midpoint of the superficial surface.
superextensionvert = [superficialverts(superproxloc,1)+superhalfx superficialverts(superproxloc,2)+superhalfy];

% Together, the midpoints of the deep and superficial boundaries define a 
% line segment that bisects the total CTF ROI into distal and proximal 
% halves. Combine the coordinates into a matrix to define this
% internal dividing boundary.
middle = [(superextensionvert(1,1)-deepextensionhalfvert(1,1))/2 (superextensionvert(1,2)-deepextensionhalfvert(1,2))/2]+...
    deepextensionhalfvert;

% Calculate the midpoint of the distal boundary using the two points that 
% define the line segment.
distalhalf = [(distalboundary(end,1)-distalboundary(1,1))/2 (distalboundary(end,2)-distalboundary(1,2))/2]+...
    distalboundary(1,:);

% Similarly, calculate the midpoint of the proximal boundary using the two 
% points that define the line segment.
proxhalf = [(proximalboundary(end,1)-proximalboundary(1,1))/2 (proximalboundary(end,2)-proximalboundary(1,2))/2]+...
    proximalboundary(1,:);

% Now, the user will approximately divide the total CTF ROI into 
% superfical and deep halves using the drawpolyline() function to connect
% the midpoint of the proximal boundary to the midpoint of the internal
% dividing boundary separating distal from proximal. Continuing distally,
% the user will connect the midpoint of the internal dividing boundary
% separating distal from proximal to the midpoint of the distal boundary
% using drawpolyline.

% Display the RGB image with all boundaries overlaid, and overlay the
% midpoints of the proximal, distal, and internal boundary dividing distal
% and proximal halves. These midpoints will appear as open yellow circles.
figure(6)
imshow(RGB); hold on
plot(boundary(:,1),boundary(:,2),'r');
plot([deepextensionhalfvert(1,1) superextensionvert(1,1)],[deepextensionhalfvert(1,2) superextensionvert(1,2)],'r');
plot(distalhalf(1,1),distalhalf(1,2),'yo');
plot(proxhalf(1,1),proxhalf(1,2),'yo');
plot(middle(1,1),middle(1,2),'yo'); hold off

% Starting at the midpoint of the proximal boundary, draw a polyline
% connecting to the midpoint of the internal boundary dividing distal from
% proximal, following the fiber orientation and attempting to stay halfway
% between the deep and superficial boundaries at all times.
proximallongdivision = drawpolyline('Color','Red');
% Extract the coordinate positions of this internal boundary.
proximalmidline = proximallongdivision.Position;

close all

% Display the image with all defined boundaries overlaid. Again, overlay
% the midpoints of the proximal, distal, and internal boundaring separating
% distal from proximal as open yellow circles. 
figure(7)
imshow(RGB); hold on
plot(boundary(:,1),boundary(:,2),'r');
plot([deepextensionhalfvert(1,1) superextensionvert(1,1)],[deepextensionhalfvert(1,2) superextensionvert(1,2)],'r');
plot(distalhalf(1,1),distalhalf(1,2),'yo');
plot(proxhalf(1,1),proxhalf(1,2),'yo');
plot(middle(1,1),middle(1,2),'yo');
plot(proximalmidline(:,1),proximalmidline(:,2),'r'); hold off

% Continuing distally, draw a polyline connecting the midpoint of the
% internal boundary dividing distal and proximal to the midpoint of the
% distal boundary, following the fiber orientation while attempting to stay
% halfway between the superficial and deep boundaries at all times. 
distallongdivision = drawpolyline('Color','Red');
% Extract the coordinates of this internal boundary.
distalmidline = distallongdivision.Position;

close all

% This procedure provides all coordinates needed to outline the total CTF
% ROI and subdivide into 4 quadrants bisecting distal from proximal and
% superficial from deep. Display the image with all boundaries overlaid,
% and save this image as a .tif to the Current Folder.
figure(8)
imshow(RGB); hold on
plot(boundary(:,1),boundary(:,2),'r');
plot([deepextensionhalfvert(1,1) superextensionvert(1,1)],[deepextensionhalfvert(1,2) superextensionvert(1,2)],'r');
plot(proximalmidline(:,1),proximalmidline(:,2),'r');
plot(distalmidline(:,1),distalmidline(:,2),'r'); hold off

% To save this image as 'filename_boundaries.tif', obtain the base filename
% using fileparts().
[folder, baseFileNameNoExt, extension] = fileparts(filename);
% Create new image filename: 'filename_boundaries.tif'
imageout = sprintf('%s%s.tif',baseFileNameNoExt,'_boundaries');
% Save current figure
saveas(gcf,imageout);
close all

% Compile coordinates defining quadrant 1 (distal deep).
quadrant1 = [distaldeephalftrimmed(2:end,:);deepextensionhalfvert;middle;distalmidline(2:end-1,:);distalhalf;distalboundary(1,:)];

% Compile coordinates defining quadrant 2 (proximal deep).
quadrant2 = [proximaldeephalftrimmed;proxhalf;proximalmidline(2:end-1,:);middle;deepextensionhalfvert];

% Compile coordinates defining quadrant 3 (distal superficial).
quadrant3 = [distalhalf;distalboundary(2,:);distalsupertrimmed(2:end,:);superextensionvert;middle;distalmidline(2:end-1,:)];

% Compile coordinates defining quadrant 4 (proximal superficial).
quadrant4 = [proximalsupertrimmed(1:end-1,:);proximalboundary(2,:);proxhalf;proximalmidline(2:end-1,:);middle;superextensionvert];

% Compile coordinates defining the distal region (quad1 & quad3).
distal = [distaldeephalftrimmed(2:end,:);deepextensionhalfvert;superextensionvert;flip(distalsupertrimmed(2:end,:));flip(distalboundary)];

% Compile coordinates defining the proximal region (quad2 & quad4).
proximal = [proximaldeephalftrimmed;proximalboundary(2,:);flip(proximalsupertrimmed(1:end-1,:));superextensionvert;deepextensionhalfvert];

% Compile coordinates defining the deep region (quad1 & quad2).
deepregion = [deepboundary(2:end,:);proxhalf;proximalmidline(2:end-1,:);middle;distalmidline(2:end-1,:);distalhalf;distalboundary(1,:)];

% Compile coordinates defining the superficial region (quad3 & quad4):
superficial = [distalboundary(2,:);superficialverts(2:end-1,:);proximalboundary(2,:);proxhalf;proximalmidline(2:end-1,:);middle;distalmidline(2:end-1,:);distalhalf];

% Lastly, use the drawpolyline() function to circle a small ROI in the
% periosteal fibrocartilage (PF) with rich GAG staining.
figure(9)
imshow(RGB);
PF = drawpolyline('Color','r');
% Extract the coordinates for this ROI
PFverts = PF.Position;
close all

% To export the ROI coordinates as a .xlsx file, compile all coordinates
% into a table. To do this, all coordinate matrices must be equivalent in
% length. First, determine the length of the largest coordinate matrix
% across ROIs. Preallocate matrices of equivalent length with NaN values,
% then insert coordinates into these dummy matrices. Excel treats NaN
% values as empty cells.
TotalROIx = NaN(max(max(length(boundary),length(deepregion)),...
    length(superficial)),1);
TotalROIx(1:length(boundary),1) = boundary(:,1);
TotalROIy = NaN(max(max(length(boundary),length(deepregion)),...
    length(superficial)),1);
TotalROIy(1:length(boundary),1) = boundary(:,2);
Quadrant1ROIx = NaN(max(max(length(boundary),length(deepregion)),...
    length(superficial)),1);
Quadrant1ROIx(1:length(quadrant1),1) = quadrant1(:,1);
Quadrant1ROIy = NaN(max(max(length(boundary),length(deepregion)),...
    length(superficial)),1);
Quadrant1ROIy(1:length(quadrant1),1) = quadrant1(:,2);
Quadrant2ROIx = NaN(max(max(length(boundary),length(deepregion)),...
    length(superficial)),1);
Quadrant2ROIx(1:length(quadrant2),1) = quadrant2(:,1);
Quadrant2ROIy = NaN(max(max(length(boundary),length(deepregion)),...
    length(superficial)),1);
Quadrant2ROIy(1:length(quadrant2),1) = quadrant2(:,2);
Quadrant3ROIx = NaN(max(max(length(boundary),length(deepregion)),...
    length(superficial)),1);
Quadrant3ROIx(1:length(quadrant3),1) = quadrant3(:,1);
Quadrant3ROIy = NaN(max(max(length(boundary),length(deepregion)),...
    length(superficial)),1);
Quadrant3ROIy(1:length(quadrant3),1) = quadrant3(:,2);
Quadrant4ROIx = NaN(max(max(length(boundary),length(deepregion)),...
    length(superficial)),1);
Quadrant4ROIx(1:length(quadrant4),1) = quadrant4(:,1);
Quadrant4ROIy = NaN(max(max(length(boundary),length(deepregion)),...
    length(superficial)),1);
Quadrant4ROIy(1:length(quadrant4),1) = quadrant4(:,2);
DistalROIx = NaN(max(max(length(boundary),length(deepregion)),...
    length(superficial)),1);
DistalROIx(1:length(distal),1) = distal(:,1);
DistalROIy = NaN(max(max(length(boundary),length(deepregion)),...
    length(superficial)),1);
DistalROIy(1:length(distal),1) = distal(:,2);
ProximalROIx = NaN(max(max(length(boundary),length(deepregion)),...
    length(superficial)),1);
ProximalROIx(1:length(proximal),1) = proximal(:,1);
ProximalROIy = NaN(max(max(length(boundary),length(deepregion)),...
    length(superficial)),1);
ProximalROIy(1:length(proximal),1) = proximal(:,2);
DeepROIx = NaN(max(max(length(boundary),length(deepregion)),...
    length(superficial)),1);
DeepROIx(1:length(deepregion),1) = deepregion(:,1);
DeepROIy = NaN(max(max(length(boundary),length(deepregion)),...
    length(superficial)),1);
DeepROIy(1:length(deepregion),1) = deepregion(:,2);
SuperficialROIx = NaN(max(max(length(boundary),length(deepregion)),...
    length(superficial)),1);
SuperficialROIx(1:length(superficial),1) = superficial(:,1);
SuperficialROIy = NaN(max(max(length(boundary),length(deepregion)),...
    length(superficial)),1);
SuperficialROIy(1:length(superficial),1) = superficial(:,2);
PFROIx = NaN(max(max(length(boundary),length(deepregion)),...
    length(superficial)),1);
PFROIx(1:length(PFverts),1) = PFverts(:,1);
PFROIy = NaN(max(max(length(boundary),length(deepregion)),...
    length(superficial)),1);
PFROIy(1:length(PFverts),1) = PFverts(:,2);

% Now, create a table of ROI coordinates.
T = table(TotalROIx,TotalROIy,Quadrant1ROIx,Quadrant1ROIy,Quadrant2ROIx,...
    Quadrant2ROIy,Quadrant3ROIx,Quadrant3ROIy,Quadrant4ROIx,Quadrant4ROIy,...
    DistalROIx,DistalROIy,ProximalROIx,ProximalROIy,DeepROIx,DeepROIy,...
    SuperficialROIx,SuperficialROIy,PFROIx,PFROIy);

% Assign column headers.
T.Properties.VariableNames = {'Total CTF (x)','Total CTF (y)','Quadrant1 (x)',...
    'Quadrant1 (y)','Quadrant2 (x)','Quadrant2 (y)','Quadrant3 (x)','Quadrant3 (y)',...
    'Quadrant4 (x)','Quadrant4 (y)','Distal (x)','Distal (y)','Proximal (x)',...
    'Proximal (y)','Deep (x)','Deep (y)','Superficial (x)','Superficial (y)'...
    'Periosteal Fibrocartilage (x)','Periosteal Fibrocartilage (y)'};

% Create a filename for the excel sheet.
fileout = sprintf('%s%s.xlsx',baseFileNameNoExt,'_boundaries');
% Write the table to an excel sheet, created in the Current Folder.
writetable(T,fileout);

% Output the coordinate matrices for each ROI compiled into a cell array
% into the MATLAB workspace.
out = {boundary,quadrant1,quadrant2,quadrant3,quadrant4,distal,proximal,deepregion,superficial,PFverts};

end