addpath 'functions'
addpath 'spotdetector'
addpath 'regu'
addpath 'l1magic-1.1/Optimization/'
addpath(genpath('functions_fforce'));

% initial bead locations -> separate image
% parameter redifinition
% GFP -> GFP
% research: unit of force -> indicate in output .csv
% L curve graph -> delete
% due date: 7/26

app = struct( ...
    ... % UI components
    'Figure', [], ...
    'LeftPanel', [], ...
    'RightPanel', [], ...
    'LoadInitialBeadImageButton', [], ...
    'LoadBeadImageButton', [], ...
    'LoadCellImageButton', [], ...
    'LoadGFPImageButton', [], ...
    'SpotDetectButton', [], ...
    'PlotArea', [], ...
    'PlotSelector', [], ...
    'Slider', [], ...
    'ROIButton', [], ...
    'TrackButton', [], ...
    'MakeImageStackButton', [], ...
    'PlotTrackButton', [], ...
    'SelectBeadsButton', [], ...
    'EstimateForceButton', [], ...
    'Toolbar', [], ...
    'SaveButton', [], ...
    'LoadButton', [], ...
    ... % Image stacks 
    'BeadImageStack', [], ...
    'CellImageStack', [], ...
    'GFPImageStack', [], ...
    ...
    'TrackRegistration', [], ...
    'ROI', struct('attempt', [], 'atrange', []), ...
    'BeadsData', [], ...
    'SelectedBeadsNo', [], ...
    'FilePath', struct('InitialBeads', [], ...
                              'Beads', [], ...
                              'Cell', [], ...
                              'GFP', []), ...
    'Parameter', struct('YMod', [], ...
                                   'poisR', [], ...
                                   'unitPixel', [], ...
                                   'thrGS', []));

app.Figure = figure('Position', [100, 100, 800, 500]);
app.LeftPanel = uipanel(app.Figure);
app.LeftPanel.Position = [.01 .01 .3 .98];
app.RightPanel = uipanel(app.Figure);
app.RightPanel.Position = [.32 .01 .67 .98];

buttonTopX = 10;
buttonTopY = 460;
buttonWidth = 210;
buttonHeight = 20;
buttonMargin = 10;

app.LoadInitialBeadImageButton = uicontrol( ...
    app.LeftPanel, ...
    'Units', 'pixels', ...
    'InnerPosition', [buttonTopX buttonTopY buttonWidth buttonHeight], ...
    'Style', 'pushbutton', ...
    'String', 'Load unstrained substrate image', ...
    'Tag', 'load initial bead image', ...
    'Callback', @loadImageButtonCallback);

app.LoadBeadImageButton = uicontrol( ...
    app.LeftPanel, ...
    'Units', 'pixels', ...
    'InnerPosition', [buttonTopX buttonTopY - 1 * (buttonHeight + buttonMargin) buttonWidth buttonHeight], ...
    'Style', 'pushbutton', ...
    'String', 'Load fluorescent bead images', ...
    'Tag', 'load bead image', ...
    'Callback', @loadImageButtonCallback);

app.LoadCellImageButton = uicontrol( ...
    app.LeftPanel, ...
    'Units', 'pixels', ...
    'InnerPosition', [buttonTopX buttonTopY - 2 * (buttonHeight + buttonMargin) buttonWidth buttonHeight], ...
    'Style', 'pushbutton', ...
    'String', 'Load bright-field images', ...
    'Tag', 'load cell image', ...
    'Callback', @loadImageButtonCallback);

app.LoadGFPImageButton = uicontrol( ...
    app.LeftPanel, ...
    'Units', 'pixels', ...
    'InnerPosition', [buttonTopX buttonTopY - 3 * (buttonHeight + buttonMargin) buttonWidth buttonHeight], ...
    'Style', 'pushbutton', ...
    'String', 'Load GFP images', ...
    'Tag', 'load gfp image', ...
    'Callback', @loadImageButtonCallback);

app.SpotDetectButton = uicontrol( ...
    app.LeftPanel, ...
    'Units', 'pixels', ...
    'InnerPosition', [buttonTopX buttonTopY - 7 * (buttonHeight + buttonMargin) buttonWidth buttonHeight], ...
    'Style', 'pushbutton', ...
    'String', 'Spot detect', ...
    'Tag', 'spot detect', ...
    'Callback', @spotDetectButtonCallBack);

app.PlotSelector = uicontrol( ...
    app.LeftPanel, ...
    'Units', 'pixels', ...
    'InnerPosition', [buttonTopX buttonTopY - 4 * (buttonHeight + buttonMargin) buttonWidth buttonHeight], ...
    'Style', 'popupmenu', ...
    'String', {'Select', 'Beads', 'Cell', 'GFP'}, ...
    'Tag', 'selector', ...
    'Callback', @plotSelectorCallBack);
    
app.Slider = uicontrol( ...
    app.LeftPanel, ...
    'Units', 'pixels', ...
    'InnerPosition', [buttonTopX buttonTopY - 5 * (buttonHeight + buttonMargin) buttonWidth buttonHeight], ...
    'Style', 'slider', ...
    'SliderStep', [0.01, 0.1], ...
    'Min', 1, ...
    'Max', 1, ...
    'Value', 1, ...
    'Tag', 'slider', ...
    'Callback', @sliderCallBack);

app.ROIButton = uicontrol( ...
    app.LeftPanel, ...
    'Units', 'pixels', ...
    'InnerPosition', [buttonTopX buttonTopY - 6 * (buttonHeight + buttonMargin) buttonWidth buttonHeight], ...
    'Style', 'pushbutton', ...
    'String', 'ROI', ...
    'Tag', 'roi', ...
    'Callback', @roiButtonCallBack);

app.PlotTrackButton = uicontrol( ...
    app.LeftPanel, ...
    'Units', 'pixels', ...
    'InnerPosition', [buttonTopX buttonTopY - 9 * (buttonHeight + buttonMargin) buttonWidth buttonHeight], ...
    'Style', 'togglebutton', ...
    'Min', 0, ...
    'Max', 1, ...
    'String', 'Plot track', ...
    'Tag', 'make image stack', ...
    'Callback', @plotTrackButtonCallBack);

app.SelectBeadsButton = uicontrol( ...
    app.LeftPanel, ...
    'Units', 'pixels', ...
    'InnerPosition', [buttonTopX buttonTopY - 10 * (buttonHeight + buttonMargin) buttonWidth buttonHeight], ...
    'Style', 'pushbutton', ...
    'String', 'Select beads', ...
    'Tag', 'select beads', ...
    'Callback', @selectBeadsButtonCallBack);

app.EstimateForceButton = uicontrol( ...
    app.LeftPanel, ...
    'Units', 'pixels', ...
    'InnerPosition', [buttonTopX buttonTopY - 11 * (buttonHeight + buttonMargin) buttonWidth buttonHeight], ...
    'Style', 'pushbutton', ...
    'String', 'Estimate force', ...
    'Tag', 'estimate force', ...
    'Callback', @estimateForceButtonCallBack);

% app.MakeImageStackButton = uicontrol( ...
%     app.LeftPanel, ...
%     'Units', 'pixels', ...
%     'InnerPosition', [buttonTopX buttonTopY - 13 * (buttonHeight + buttonMargin) buttonWidth buttonHeight], ...
%     'Style', 'pushbutton', ...
%     'String', 'make image stack', ...
%     'Tag', 'make image stack', ...
%     'Callback', @imagePreprocessButtonCallback);

app.Toolbar = uitoolbar(app.Figure);

pathToIcon = fullfile('LuckyIconBasicSet/', 'ico/', '256/', 'Normal/', '16x16-256/');
app.SaveButton = uipushtool(app.Toolbar);
[img, map] = imread(fullfile(pathToIcon, 'Save as.ico'));
img = ind2rgb(img, map);
app.SaveButton.CData = img;
app.SaveButton.ClickedCallback = @saveAppDataGUI;

app.LoadButton = uipushtool(app.Toolbar);
[img, map] = imread(fullfile(pathToIcon, 'Open.ico'));
img = ind2rgb(img, map);
app.LoadButton.CData = img;
app.LoadButton.ClickedCallback = @loadAppDataGUI;

app.PlotArea = subplot(1, 1, 1);
app.PlotArea.Parent = app.RightPanel;

% p = fullfile('120118 beads/RNAi/9/0/beads', 'sstack.tif');
% app.ImageStack = loadImageStack(p);
% imagesc(app.PlotArea, app.ImageStack(:, :, :, 1));

% showAsMovie(app.ImageStack);

function loadImageButtonCallback(src, ~)
app = evalin('base', 'app');
tag = src.Tag;
switch tag
    case {'load initial bead image'}
        [img, p] = getImageStack('Select unstrained substrate image');
        app.BeadImageStack = img;
        app.FilePath.InitialBeads = p;
        figure; imshow(img);
    case {'load bead image'}
        [stack, p] = getImageStack('Select fluorescent bead images');
        if isempty(app.BeadImageStack)
            input('please load unstrained substrate image in advance');
        else
            app.BeadImageStack = cat(4, app.BeadImageStack, stack);
        end
        app.FilePath.Beads = p;
        showAsMovie(app.BeadImageStack);
    case {'load cell image'}
        [stack, p] = getImageStack('Select bright-field images');
        app.CellImageStack = stack;
        app.FilePath.Cell = p;
        showAsMovie(stack);
    case {'load gfp image'}
        [stack, p] = getImageStack('Select GFP images');
        app.GFPImageStack = stack;
        app.FilePath.GFP = p;
        showAsMovie(stack);
end
assignin('base', 'app', app);
end

function spotDetectButtonCallBack(src, ~)
app = evalin('base', 'app');
[B, E, C, reg0] = traceBeads_gf(app, [0, 0]);
% app.BeadImageStack = B0;
% app.CellImageStack = C0;
% app.GFPImageStack = E0;
app.TrackRegistration = reg0;

% B = app.BeadImageStack;
% E = app.GFPImageStack;
% C = app.CellImageStack;
attention = app.ROI.attempt;
aterange = app.ROI.atrange;

degs= -[...
   180;...
   % 60;...
    ]./360*(2*pi);

th_GSarea = inputdlg('Input threshold', '', [1 35], {'0'});
th_GSarea = str2double(th_GSarea{1});
app.Parameter.thrGS = th_GSarea;

th_GSarea = [...
    th_GSarea;...
    %300;...
    ];

beadsdata = catchBeads_gf(B, E, C, attention, aterange, degs(1), th_GSarea(1), false, 50);
app.BeadsData = beadsdata;
assignin('base', 'app', app);
end

function trackButtonCallBack(src, ~)
app = evalin('base', 'app');
B = app.BeadImageStack;
E = app.GFPImageStack;
C = app.CellImageStack;
attention = app.ROI.attempt;
aterange = app.ROI.atrange;

degs= -[...
   180;...
   % 60;...
    ]./360*(2*pi);

% if isempty(app.Parameter.thrGS)
%     th_GSarea = inputdlg('Input threshold', '', [1 35], {'0'});
%     th_GSarea = str2double(th_GSarea{1});
%     app.Parameter.thrGS = th_GSarea;
% else
%     th_GSarea = app.Parameter.thrGS;
% end

th_GSarea = inputdlg('Input threshold', '', [1 35], {'0'});
th_GSarea = str2double(th_GSarea{1});
app.Parameter.thrGS = th_GSarea;

th_GSarea = [...
    th_GSarea;...
    %300;...
    ];

beadsdata = catchBeads_gf(B, E, C, attention, aterange, degs(1), th_GSarea(1), false, 50);
app.BeadsData = beadsdata;
assignin('base', 'app', app);
end

function plotSelectorCallBack(src, ~)
app = evalin('base', 'app');
val = src.Value;
if val == 1
    return;
end

slider = app.Slider;
switch val
    case 2
        slider.Max = stackLen(app.BeadImageStack);
    case 3
        slider.Max = stackLen(app.CellImageStack);
    case 4
        slider.Max = stackLen(app.CellImageStack);
end
slider.SliderStep = [1 1] / (slider.Max - slider.Min);
slider.Value = 1;
showOneImage();

% disp(slider);
% uicontrol(slider);

assignin('base', 'app', app);
end

function sliderCallBack(src, ~)
showOneImage();
end

function roiButtonCallBack(src, ~)
app = evalin('base', 'app');
[x, y] = ginput(2);
x = round(x);
y = round(y);
attempt = [mean(x), mean(y)]; attempt = round(attempt);
atrange = [abs(x(1) - x(2)) / 2 abs(y(1) - y(2)) / 2]; atrange = round(atrange);
app.ROI.attempt = attempt;
app.ROI.atrange = atrange;

showOneImage();
hold(app.PlotArea, 'on');
plot(app.PlotArea, [x(1), x(1)], [y(1) y(2)], 'r', 'LineWidth', 2);
plot(app.PlotArea, [x(1), x(2)], [y(1) y(1)], 'r', 'LineWidth', 2);
plot(app.PlotArea, [x(2), x(2)], [y(1) y(2)], 'r', 'LineWidth', 2);
plot(app.PlotArea, [x(1), x(2)], [y(2) y(2)], 'r', 'LineWidth', 2);
hold(app.PlotArea, 'off');

assignin('base', 'app', app);
end

function plotTrackButtonCallBack(src, ~)
app = evalin("base", "app");
showOneImage();
end

function imagePreprocessButtonCallback(src, ~)
app = evalin('base', 'app');
tag = src.Tag;
switch tag
    case {'make image stack'}
        stackImagesGUI();
end
end

function selectBeadsButtonCallBack(src, ~)
app = evalin('base', 'app');
x = []; y = [];
while true
    temp = ginput(1);
    if isempty(temp)
        break;
    end
    x(end+1) = temp(1);
    y(end+1) = temp(2);
end
ps = polyshape(x, y);
x1 = 0;
x2 = 2 * app.ROI.atrange(1);
y1 = 0;
y2 = 2 * app.ROI.atrange(2);
psq = polyshape([x1-1 x1-1 x2+1 x2+1], [y2+1 y1-1 y1-1 y2+1]);
ps = intersect(ps, psq);
sliderVal = round(app.Slider.Value);
TFin = isinterior(ps, ...
    app.BeadsData.coords(:, 1, 1), ...
    app.BeadsData.coords(:, 2, 1));
app.SelectedBeadsNo = find(TFin);

assignin('base', 'app', app);
showOneImage();
end

function estimateForceButtonCallBack(src, ~)
app = evalin('base', 'app');
useprocess = zeros(size(app.BeadsData.coords, 1), 1);
useprocess(app.SelectedBeadsNo) = 1;

unitpixel = inputdlg('Input pixel size [um/pixel]', '', [1 35], {'1'});
unitpixel = str2double(unitpixel{1});
app.Parameter.unitPixel = unitpixel * 1e-6;

yModule = inputdlg("Input Young's modules [Pa]", '', [1 35], {'1'});
yModule = str2double(yModule{1});
app.Parameter.YMod = yModule;

Pratio = inputdlg("Input Poisson's ratio", '', [1 35], {'1'});
Pratio = str2double(Pratio{1});
app.Parameter.poisR = Pratio;

[app.BeadsData, ~] = FestProtType(app.BeadsData, unitpixel, yModule, Pratio, useprocess, []);

fname_force_xls = '';
while true
    fname_force_xls = inputdlg({'Enter file name:'}, '', [1 35], {''});
    if ~isempty(fname_force_xls)
        break;
    end
end

fx = reshape(mean(app.BeadsData.fx, 1), [], 1);
fy = reshape(mean(app.BeadsData.fy, 1), [], 1);
mags = sqrt(fx.^2 + fy.^2);

force = horzcat(fx, fy, mags);
force = force / unitpixel ^ 3;
force = table(force(:, 1), force(:, 2), force(:, 3), 'VariableNames', {'X-component [Pa]', 'Y-component [Pa]', 'Magnitude of traction force [Pa]'});
writetable(force, strcat(fname_force_xls{1}, '.xls'));

assignin('base', 'app', app);
end

function showOneImage()
app = evalin('base', 'app');
if isempty(app.BeadImageStack) || isempty(app.CellImageStack) || isempty(app.GFPImageStack)
    return;
end
if app.PlotTrackButton.Value == 0 || isempty(app.ROI)
    xind = 1:size(app.BeadImageStack, 1);
    yind = 1:size(app.BeadImageStack, 1);
elseif ~isempty(app.ROI) && app.PlotTrackButton.Value == 1
    roi = app.ROI;
    xind = roi.attempt(1) - roi.atrange(1):roi.attempt(1) + roi.atrange(1);
    yind = roi.attempt(2) - roi.atrange(2):roi.attempt(2) + roi.atrange(2);
else
    xind = 1:size(app.BeadImageStack, 1);
    yind = 1:size(app.BeadImageStack, 1);
end

selectorVal = app.PlotSelector.Value;
sliderVal = round(app.Slider.Value);
txt = sprintf('%d/', sliderVal);

cla(app.PlotArea);
hold(app.PlotArea, 'on');
switch selectorVal
    case 2
        image(app.PlotArea, app.BeadImageStack(yind, xind, :, sliderVal));
        txt = sprintf([txt, '%d'], stackLen(app.BeadImageStack));
    case 3
        imagesc(app.PlotArea, app.CellImageStack(yind, xind, :, sliderVal));
        txt = sprintf([txt, '%d'], stackLen(app.CellImageStack));
    case 4
        imagesc(app.PlotArea, app.GFPImageStack(yind, xind, :, sliderVal));
        txt = sprintf([txt, '%d'], stackLen(app.GFPImageStack));
end

if app.PlotTrackButton.Value == 1 && ~isempty(app.ROI)
    index = sliderVal;
    switch app.PlotSelector.Value
        case{3, 4}
            index = index + 1;
    end
    xcoords = app.BeadsData.coords(:, 1, index);
    ycoords = app.BeadsData.coords(:, 2, index);
    plot(xcoords, ycoords, "ro", "MarkerSize", 5, "MarkerFaceColor", "w");
    
    if ~isempty(app.SelectedBeadsNo)
        xcoords = app.BeadsData.coords(app.SelectedBeadsNo, 1, index);
        ycoords = app.BeadsData.coords(app.SelectedBeadsNo, 2, index);
        plot(xcoords, ycoords, "o", "MarkerSize", 6, "MarkerFaceColor", "r", "MarkerEdgeColor", "k");
    end
end

axis(app.PlotArea, 'equal');
axis(app.PlotArea, 'tight');
axis(app.PlotArea, 'ij');
title(app.PlotArea, txt);
hold(app.PlotArea, 'off');
end

function [stack, p] = getImageStack(msg, grayscaling)
if nargin < 1
    msg = 'Select image';
end
if nargin < 2
    grayscaling = false;
end
[f, p] = uigetfile({ ...
    '*.tif;*.tiff;*.TIF;*.TIFF', 'Tiff image'; ...
    '*.*', 'All Files'}, msg);

if f == 0
    return;
end

[~, name, ext] = fileparts(f);
p = [p, name, ext];
stack = loadImageStack(p, grayscaling);
end

function stack = loadImageStack(p, grayscaling)
if nargin < 2
    grayscaling = false;
end

i = 1;
stack = [];
while true
    try
        tmp = imread(p, i);
        if grayscaling
            tmp = rgb2gray(tmp);
            stack = cat(4, stack, tmp);
        else
            stack = cat(4, stack, tmp);
        end
        i = i + 1;
    catch E
        break;
    end
end
end

function len = stackLen(stack)
dim = length(size(stack));
len = size(stack, dim);
end

function fh = showAsMovie(stack, isgray)
if nargin < 2
    isgray = false;
end
len = stackLen(stack);
F(len) = struct('cdata', [], 'colormap', []);
for i=1:len
    if isgray
        [X, cmap] = gray2ind(stack(:, :, i), 64);
    else
        [X, cmap] = rgb2ind(stack(:, :, :, i), 255);
    end
    
    F(i) = im2frame(X, cmap);
end
fh = figure;
movie(fh, F, 1, 10);
end

function stackImagesGUI()
[f, p] = uigetfile({ ...
    '*.tif;*.tiff;*.TIF;*.TIFF', 'TIFF Image'; ...
    '*.*', 'All Files'}, 'Select image files', 'MultiSelect', "on");
if ~iscell(f)
    return;
end

answer = inputdlg("Save As", "");
if isempty(answer)
    return;
elseif isempty(strfind(answer{1}, "."))
    answer = answer{1} + ".tif";
else
    answer = answer{1};
end

N = length(f);
imwrite(imread(fullfile(p, f{1})), fullfile(p, answer), "tif");
for i=2:N
    imwrite(imread(fullfile(p, f{i})), fullfile(p, answer), "tif", "WriteMode", "append");
end
end

function saveAppDataGUI(src, ~)
app = evalin("base", "app");
uisave("app",  "app");
end

function loadAppDataGUI(src, ~)
[f, p] = uigetfile({ ...
    '*.mat', 'MAT File'; ...
    '*.*', 'All Files'}, 'Select mat file');
if f == 0
    return;
end
loadAppData(fullfile(p, f));
end

function loadAppData(p)
m = load(p);
if ~isfield(m, "app")
    return;
end
close(m.app.Figure);

app = evalin("base", "app");
if isfield(m.app, "FilePath")
    app.FilePath = m.app.FilePath;
    app.BeadImageStack = loadImageStack(app.FilePath.InitialBeads);
    app.BeadImageStack = cat(4, app.BeadImageStack, loadImageStack(app.FilePath.Beads));
    app.CellImageStack = loadImageStack(app.FilePath.Cell);
    app.GFPImageStack = loadImageStack(app.FilePath.GFP);
end
if isfield(m.app, "ROI")
    app.ROI = m.app.ROI;
end
if isfield(m.app, "TrackRegistration")
    app.TrackRegistration = m.app.TrackRegistration;
end
if isfield(m.app, "BeadsData")
    app.BeadsData = m.app.BeadsData;
end
if isfield(m.app, "SelectedBeadsNo")
    app.SelectedBeadsNo = m.app.SelectedBeadsNo;
end
assignin("base", "app", app);
end