function [filledMask] = FillExterior(mask, dimension, display)
% FillExterior Perform an iterative 2D flood fill operation from the
% corners of the frame in specified dimension
% =========================================================================
% AUTHOR       Zachary Ip
% CONTACT      ipzach@uw.edu
% INSTITUTION  University of Washington
% DATE        25th November 2019
% 
% INPUTS mask - 3D logical - binary mask to be filled
%        dimension - int, the dimension to fill data
%        display - logical - toggle visualization of fill function on or
%        off.
%   
% OUTPUTS filledMask - 3D logical - filled mask
% =========================================================================
% Ensure dimension is a valid value
valid = [1 2 3];
if ~ismember(dimension, valid)
    error('Please enter an integer between 1 and 3')
end
% initialize filledMask
filledMask = zeros(size(mask));

switch dimension
    case 1
        % Get a reference frame to determine seed points
        frame = squeeze(mask(1,:,:));
        topLeft = [size(frame,1)-2, 2];
        topRight = [size(frame,1)-2, size(frame,2)-2];
        bottomLeft = [2, 2];
        bottomRight = [2, size(frame,2)-2];
        
        seed = [topLeft; topRight; bottomLeft; bottomRight];
        % Fill mask at each seed point
        for i = 1:size(mask,1)
            filledMask(i,:,:) = imfill(logical(squeeze(mask(i,:,:))),seed);
            if display
                subplot(1,2,1)
                pcolor(flipud(squeeze(mask(i,:,:)))), shading interp, colormap(bone), axis off
                subplot(1,2,2)
                pcolor(flipud(squeeze(filledMask(i,:,:)))), shading interp, colormap(bone), axis off
                drawnow
            end % if
        end % i
    case 2
        frame = squeeze(mask(:,1,:));
        topLeft = [size(frame,1)-2, 2];
        topRight = [size(frame,1)-2, size(frame,2)-2];
        bottomLeft = [2, 2];
        bottomRight = [2, size(frame,2)-2];
        
        seed = [topLeft; topRight; bottomLeft; bottomRight];
        
        for i = 1:size(mask,2)
            filledMask(:,i,:) = imfill(logical(squeeze(mask(:,i,:))),seed);
            if display
                subplot(1,2,1)
                pcolor(flipud(squeeze(mask(:,i,:)))), shading interp, colormap(bone), axis off
                subplot(1,2,2)
                pcolor(flipud(squeeze(filledMask(:,i,:)))), shading interp, colormap(bone), axis off
                drawnow
            end % if
        end % i
    case 3
        frame = squeeze(mask(:,:,1));
        topLeft = [size(frame,1)-2, 2];
        topRight = [size(frame,1)-2, size(frame,2)-2];
        bottomLeft = [2, 2];
        bottomRight = [2, size(frame,2)-2];
        
        seed = [topLeft; topRight; bottomLeft; bottomRight];
        
        for i = 1:size(mask,3)
                filledMask(:,:,i) = imfill(logical(squeeze(mask(:,:,i))),seed);
            if display
                subplot(1,2,1)
                pcolor(flipud(squeeze(mask(:,:,i)))), shading interp, colormap(bone), axis off
                subplot(1,2,2)
                pcolor(flipud(squeeze(filledMask(:,:,i)))), shading interp, colormap(bone), axis off
                drawnow
            end % if
        end % i
end % switch
end % fuction