"""
-Spirometer data analysis example-
Description: Contains only the basics for the analysis of spirometer data
For a more comprehensive package refer to: https://gitlab.com/Rickdkk/worklab
Author:     Rick de Klerk
Contact:    r.de.klerk@umcg.nl
Company:    University Medical Center Groningen
License:    GNU GPLv3.0
Date:       27/06/2019
"""
import datetime
import pandas as pd
import numpy as np
import matplotlib.pyplot as plt


def load_spiro(filename: str) -> pd.DataFrame:
    """Loads COSMED spirometer data from excel file

    :param filename: full file path or file in existing path from COSMED spirometer
    :return: Spiro data in pandas dataframe
    """
    data = pd.read_excel(filename, skiprows=[1, 2], usecols="J:XX")
    data["time"] = data.apply(lambda row: dt_to_s(row["t"]), axis=1)  # hh:mm:ss to s
    data["power"] = data["EEm"] * 4184 / 60  # added power (kcal/min to J/s)
    data["weights"] = np.insert(np.diff(data["time"]), 0, 0)  # used for calculating weighted average
    data["HR"] = np.zeros(len(data)) if "HR" not in data else data["HR"]  # can be missing when sensor is not detected
    data = data[data["time"] > 0]  # remove "missing" data
    return data[["time", "Rf", "HR", "power", "VO2", "VCO2", "weights"]]


def dt_to_s(dt):
    if isinstance(dt, datetime.time):
        return (dt.hour * 60 + dt.minute) * 60 + dt.second
    else:
        h, m, s = dt.split(":")
        return int(h) * 3600 + int(m) * 60 + int(s)


def calc_weighted_average(dataframe: pd.DataFrame, weights: pd.Series) -> pd.Series:
    return dataframe.apply(lambda col: np.average(col, weights=weights), axis=0)


def main():
    test_data = load_spiro("spirometer_example_data.xlsx")
    print(calc_weighted_average(test_data[["HR", "power"]], weights=test_data["weights"]))

    fig, axes = plt.subplots(2, 1, sharex="all")
    axes[0].plot(test_data["time"], test_data["HR"])
    axes[1].plot(test_data["time"], test_data["power"])

    axes[1].set_xlabel("Time (s)")
    axes[0].set_ylabel("Heart rate (bpm)")
    axes[1].set_ylabel("Internal power (W)")
    plt.autoscale(tight=True)
    fig.align_ylabels()
    plt.show()


if __name__ == "__main__":
    main()
