function FilterMatData(dir_name, file_struct, f_subsample, f_hpf_extract, verbose)
%% %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% This function acts on the data extracted using the ExtractAbfData 
% function. It filters the subsampled data with a high pass filter in order
% to further use in the detection and characterisation of depolarisation 
% events.
%
% dir_name = a string defining the path of the folder where the input data 
% is and the output data should be saved.
% file_struct = file structure with the datasets from the desired session:
%
% file_struct(file_number, feature_number) = [session_number recording_number
% baseline/response control/target dpf errors]
% 
% Note that:
% session_number = number of the session - should be present in the
% folder name;
% recording_number = number of the recording - should be present in the
% last digits of the recording name;
% baseline/response = 0 if the trace is recorded only during spontaneous
% activity; 1 if trace contains response to PTZ 
% control/target = 0 if the trace is recorded from a Mismatch control fish;
% 1 if trace is recorded from a DEPDC5 animal
%
% f_subsample [Hz] = the sampling frequency of the signal
% f_hpf_extract [Hz] = the value of the cutoff frequency for the high pass
% filter applied for the DETECTION signal (V_extract)
% verbose = flag to select wether to plot intermediate computations (1) or 
% not (0)
%% %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

% find a list of files in the source folder
list = dir(dir_name);

if length(list)>2
    if(verbose)
        figure;
    end
    if file_struct < 0
        file_struct = 0:length(list)-3;
    end
    filename = list(3,1).name;
    clear list;
     
    % compute hpf parameters
    [b_hfp_extract,a_hfp_extract] = butter(1, f_hpf_extract*2/f_subsample, 'low');

    % iterate through datasets
    for i_datasets=1:length(file_struct)
     
        % construct file name
        if(file_struct(i_datasets))<10
            filename = [filename(1:7) num2str(file_struct(i_datasets))];
        else
            filename = [filename(1:6) num2str(file_struct(i_datasets))];
        end
        if exist([dir_name '\' filename '_raw.mat' ], 'file')
            % load file containing raw trace
            load([dir_name '\' filename '_raw' ]);
            
            % filter data: 
            % ANALYSIS signal for analysis - detrend and bring to 0      
            V_filt = detrend(V_subsample);
            V_filt = V_filt - V_filt(1);
            V_filt_analyze = V_filt;

            % DETECTION signal for event detection - subtract the lpf
            % version from the ANALYSIS signal in order to obtain a hpf. 
            V_subtract_extract = filtfilt(b_hfp_extract,a_hfp_extract, V_filt);
            V_filt_extract = V_filt-V_subtract_extract;
            clear V_subsample V_filt V_subtract_extract V_subtract_analyze;

            % plotting (if verbose active)
            if(verbose)
                time = 0:1/f_subsample:(length(V_filt_extract)-1)/f_subsample;
                subplot(length(file_struct), 1, i_datasets)
                plot(time, V_filt_analyze, 'b');
                hold on;
                plot(time, V_filt_extract, 'r');
                title(filename);
            end

            % save data
            save( [dir_name '\' filename '_filt'], 'V_filt_analyze', 'V_filt_extract');
        end
    end
end
