function ExtractAbfData(folder_id, dir_save, file_struct, f_lpf, f_subsample, verbose)
%% %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% This function extracts data from .abf files, applies and anti-alias
% filter, subsamples the data and saves it in the desired location.
%
% folder_id = a string defining the path of the folder where the data is
% located.
% dir_save = a string defining the path of the folder where the data should
% be saved.
% file_struct = file structure with the datasets from the desired session:
%
% file_struct(file_number, feature_number) = [session_number recording_number
% baseline/response control/target dpf errors]
% 
% Note that:
% session_number = number of the session - should be present in the
% folder name;
% recording_number = number of the recording - should be present in the
% last digits of the recording name;
% baseline/response = 0 if the trace is recorded only during spontaneous
% activity; 1 if trace contains response to PTZ 
% control/target = 0 if the trace is recorded from a Mismatch control fish;
% 1 if trace is recorded from a DEPDC5 animal
%
% f_lpf [Hz] = the value of the cutoff frequency for the anti-aliasing low 
% pass filter
% f_subsample [Hz] = the desired subsaplming frequency
% verbose = flag to select wether to plot intermediate computations (1) or 
% not (0)
%% %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

% find a list of files in the source folder
list = dir(folder_id);
if file_struct < 0
    file_struct = 0:length(list)-3;
end
filename_base = list(3,1).name;

% prepare a figure (if verbose active)
if(verbose)
    figure;
end

% iterate through datasets
for i_datasets=1:length(file_struct)

    % construct filename
    if(file_struct(i_datasets))<10
        filename = [filename_base(1:7) num2str(file_struct(i_datasets))];
    else
        filename = [filename_base(1:6) num2str(file_struct(i_datasets))];
    end
    
    % if filename exists
    if exist([folder_id '\' filename '.abf' ], 'file')
        % load data and information from .abf file
        [d,si,h] = abfload([folder_id '\' filename '.abf' ]);
        % compute sampling frequency in Hz from sampling interval in us 
        f_sampling = 1/si*1000000;
        data = d(:,1);
        clear d si h;
        
        % low pass data for anti-aliasing
        [b_lfp,a_lfp] = butter(5, f_lpf*2/f_sampling, 'low');
        V_filt = filtfilt(b_lfp,a_lfp, data);

        % subsample data
        i_subsample = 1:round(f_sampling/f_subsample):length(V_filt);
        V_subsample = V_filt(i_subsample);
        clear data;

        % plotting (if verbose active)
        if(verbose)
            time = 0:1/f_subsample:(length(V_subsample)-1)/f_subsample;
            subplot(length(file_struct), 1, i_datasets)
            plot(V_subsample);
            plot(time, V_subsample);
            title(list(file_struct(i_datasets)).name);
        end

        % save data
        mkdir(dir_save);
        save([dir_save '\' filename '_raw.mat'], 'V_subsample');
    end
end
