clc
clear
close all

format long G;  % 15 signficant digits

% This code calculates the Sequence of Physical Processes (SPP) metrics for
% a given dataset. These metrics are outlined in detail in the paper:
% "Rogers SA (2017) In Search of Physical Meaning: Defining Transient
% Parameters for Nonlinear Viscoelasticity. Rheol Acta 56:501-525"

% The procedure reads data from txt file and calculates and outputs
% the results based on both Fourier transform (SPPbasic_fourier_v1) and
% numerical differentiation (SPPbasic_numerical_v1) analysis methods.
%
% For the following code,  fopen('sample_file1.txt','rt') means to open 
% "sample_file1.txt" and the software is only allowed to "read" it.
% The default settings here is designed for format as following,
%
%  1st column   2nd column  3rd column  4th Column
%  Time(1)      Strain(1)   Rate(1)     Stress(1)
%  Time(1)      Strain(1)   Rate(1)     Stress(1)
%  Time(1)      Strain(1)   Rate(1)     Stress(1)
%  Time(1)      Strain(1)   Rate(1)     Stress(1)
%  Time(1)      Strain(1)   Rate(1)     Stress(1)
%  ...          ...         ...         ...

% The code requires that there be an even number of data points in each
% period of oscilation, and that the number of periods of oscilation is an
% integer. These allow for the computation of a fourier series fitting the
% input data. Additionally, please remove any header rows from the data in
% order to allow the code to extract the data properly.

% The code assumes the following units: Time (s), Strain (-), Rate (1/s),
% and Stress (Pa). If the units in the input data are different, specify
% the conversion factors in the variable 'run_conv' (details below)

% SPPplus version #1.1
% Finalized on Aug. 10th 2018 by Johnny Ching-Wei Lee, Gavin Donley and Simon Rogers @UIUC

%=======================User-defined variables=============================

filename='0010.txt'; % Input data file name (see restrictions on
                                %data format listed above)

run_state = [1,0]; %Determines what calculations to run.
    %First index denotes whether a fourier series will be used
        %if 1, SPPbasic_fourier_v1 will be run
    %Second index denotes whether a numerical differentiation will be used
        %if 1, SPPbasic_numerical_v1 will be run

print_state = [0,0];
    %First index denotes whether to print extended data (e.g. frenet-serret
    %frame components),
        %if 0, only the standard SPP data file will be produced
            %(includes waveforms, time dependent moduli and moduli rates)
            %(ex. sample_file1_SPP_LAOS_FOURIER.txt)
        %if 1, second extended data file will be produced
            %(includes components of Tangent, Normal, & Binormal vectors)
            %(ex. sample_file1_SPP_LAOS_FOURIER_TNBFRAME.txt)
    %Second index denotes whether to save figures of the SPP metrics
        %if 0, figures will be displayed but not saved
        %if 1, all figures will be saved as image files
            %(ex. sample_file1_SPP_LAOS_FOURIER_PLOT.jpg)
            %The exact figures produced will depend on the methods chosen
            %for analasis
    
run_conv = [1,1,1,1]; % Conversion factors required to put the data in
    %the assumed units: Time (s), Strain (-), Rate (1/s), and Stress (Pa),
    %with each index corresponding to the units of that specific column.
    %Conversion factors will multiply the data as follows:
    %   (data in file)*(conversion factor)=(data in correct units)

% Note: The following variables have to be specified by the user to
% implement a fourier series (function: SPPplus_fourier_v1)

omega=10; %frequency of oscilation, in units of rad/s

M=31; % The number of harmonics to be used in reconstruction of stress.
        %(Must be an odd integer)

p=1; % Total number of periods of measuring time, which we have to know in
      % order to do FT. (Must be a positive integer)

% Note: The following variables have to be specified by the user to
% implement a numerical differentiation (function: SPPplus_numerical_v1)

k=8; % Step size for numerical differentiation, default to be 1. (Must be
        %a positive integer)

num_mode=1; %method of numerical differentiation
            %0 = "standard" (does not make assumtions abot the form of the
                %data, uses forward/backward difference at ends and
                %centered derivative elsewhere)
            %1 = "looped" (assumes steady state oscilatory, uses centered
                %difference everywhere by connecting the data in a loop)

%=======================Run the SPP analysis===============================

%Read and extract data from file
[time_wave,resp_wave,L] = SPPplus_read_v1(filename,run_conv);

%Run Fourier Series analysis
if run_state(1) == 1
    SPPplus_fourier_v1(time_wave,resp_wave,L,omega,M,p,print_state,...
        filename);
end

%Run Numerical Differentiation analysis
if run_state(2) == 1
    SPPplus_numerical_v1(time_wave,resp_wave,L,k,num_mode,print_state,...
        filename);
end

%Thank you for using SPPplus.
