% Code associated with journal article 'Measuring and modeling contractile
% drying in human stratum corneum' by X. Liu and G. K. German for the Journal
% of Visualized Experiments 
% August 15 2016
%%
% Code to process PIV drying displacements, create displacement profiles
% and fit profiles with model.
%%
% This code processes grid positions and displacaments provided in the
% example results file 'd'.
load d
% This function file decomposes displacments into radial and
% azimuthal components, then azimuthally averages the data.
d = azim_av(d); % Note positions and displacements should be in pixels
d.R0 =d.r_inc(d.rad_d==max(d.rad_d)); % Establishes the peak radial displacement at the sample edge
calib = 6.45e-6 % m/pix (This calibration will convert pixel displacements into meters)
d = structfun(@(x) ( x*calib ), d, 'UniformOutput', false) % Calibrate PIV results
subplot(1,2,1); quiver(d.x,d.y,-d.dx,-d.dy);
set(gca, 'XTick', []),set(gca, 'YTick', [])
axis tight;axis square;title('Drying Displacements');
subplot (1,2,2); plot(d.r_inc/d.R0, d.rad_d,'r'); % Note: Positive values of u_r correspond to contractile displacements.
hold on;plot(d.r_inc/d.R0, d.azi_d,'b--');
axis([0 1.5 min(d.rad_d) max(d.rad_d)])
legend('u_{r}','u_{\theta}','Location','NorthWest');
ylabel('u_{r}, u_{\theta} [m]');xlabel('r/R [-]');
axis square;
%% 
% This code fits the azimuthally averaged radial displacement profiles to
% Eq. (1). using a minimum least squares approach and reports values of Esc and
% Psc

[d.afit,d.bfit,d.mod_r,d.mod_u] = Model_fit(d.r_inc,d.rad_d,d.R0,0.4,400,0);

plot(d.r_inc/d.R0, d.rad_d,'r');hold on
plot(d.mod_r/d.R0,d.mod_u,'b-')
legend('Measured radial displacement profile','Fit using Eq. (1)','Location','SouthEast')
axis([0 1.5 min(d.rad_d) max(d.rad_d)])
ylabel('u_{r}, u_{\theta} [m]');xlabel('r/R [-]');
axis([0 1.5 min(d.rad_d) max(d.rad_d)])
axis square;

Hsc=9.95e-6;% Measured SC thickness in units of meters
h=35.5e-6;% Elastoimer substrate thickness in units of meters
nu_sc=0.4; % SC Poisson's ratio
nu=0.5;% Elastomer substrate Poisson's ratio
E=16000; % Elastomer substrate Elastic modulus in units of Pa
Esc=E*(d.R0^2)*(1+nu_sc)*(1-(2*nu_sc))/(2*h*Hsc*(1+nu)*(1-nu_sc)*(d.bfit^2)) %SC Elastic Modulus in units of Pa
Psc=Esc*d.afit*(1-nu_sc)/((1+nu_sc)*(1-(2*nu_sc))) % Drying Stress in units of Pa
