function [ duration ] = time2line( data2, fn, filepath )
%time2line reads in data or excel sheel to convert time points to dashes
%and separate the flies by genotypes. It can output duration if needed.
%If genotypes are provided, they are sorted by number of bouts after 2.5 
%hours and color-coded.

%   [ duration ] = time2line( data2, fn, filepath )
%   [ duration ] = time2line( data2 ) treats the timepoints in the data2
%   matrix as from the same genotype
%   [ duration ] = time2line( [], fn, filepath ) looks for the data using
%   the file specified by fullfile(filepath,fn)
%   [ duration ] = time2line( ) uses a gui to guide the user to find the
%   excel file


% If no inputs are provided, use gui to get the file
if nargin < 1
    % Read in data
    [fn, filepath] = uigetfile({'*.xlsx';'*.xls'},...
        'Select the excel file for analysis');
    [data,textdata] = xlsread(fullfile(filepath,fn));
    
    % Throw out the first 2 columns of numbers, which are currently
    % reserved for Chamber numbers and total number of matings for each
    % fly. This number can be changed.
    n_col_throw = 2;
    data2 = data(:,(n_col_throw + 1):end);
end

% If no data but the file path is provided, get the data using the path
if isempty(data2)
    % Read in data
    [data,textdata] = xlsread(fullfile(filepath,fn));
    
    % Throw out the first 2 columns of numbers, which are currently
    % reserved for Chamber numbers and total number of matings for each
    % fly. This number can be changed.
    n_col_throw = 2;
    data2 = data(:,(n_col_throw + 1):end);
end


% Find out how many rows of data are provided
nsamples = size(data2,1);

% Find out the maximal number of bouts in the data
nmaxbouts = size(data2,2)/2;

% Initiate the duration matrix
duration = zeros(nsamples,nmaxbouts);

% If there is text data, use it to get the genotypes
if exist('textdata', 'var')
    % In case text contains column titles
    nrows2start_txt = size(textdata,1) - size(data2) + 1;
    
    % Get the genotypes column
    genos = textdata(nrows2start_txt:(nrows2start_txt+nsamples-1), 2);

    % Find out how many are unique and their indices
    [genos_unique, ~, genoinx] = unique(genos);
    
    % calculate the number of bouts after 2.5 hours
    nbouts25 = sum(data2(:,1:2:end-1) > 150,2);
    
    % Sort by bouts after 2.5 hours
    geno_nbouts25 = zeros(length(genos_unique), 1);
    
    % Calculate the number of matings after 2.5 hours for each genotype
    for i = 1 : length(genos_unique)
        geno_nbouts25(i) = mean(nbouts25(genoinx == i));
    end
    
    % Find the order
    [~, genoorder] = sort(geno_nbouts25,'descend');
    
    % Initiate new row order
    roworder = zeros(nsamples,1);
    genoinx2 = zeros(nsamples,1);
    currentrow = 1;
    
    % Write new row order
    for i = 1 : length(genos_unique)
        % row index
        rowinx = genoorder(i);
        
        % Find the rows for this geno
        rowsforgeno = find(genoinx == rowinx);
        
        % number of rows for this geno
        nrowsforgeno = length(rowsforgeno);
        
        % Write in the rows
        roworder(currentrow : currentrow + nrowsforgeno - 1)=...
            rowsforgeno;
        
        % Write in the new geno index
        genoinx2(currentrow : currentrow + nrowsforgeno - 1)=...
            rowinx;
        
        % Update current row
        currentrow = currentrow + nrowsforgeno;
    end
    
    % Reorder rows
    data2 = data2(roworder,:);
    
    
else
    % If no text data are available, set all data to be from the same geno
    genos_unique = 1;
    genoinx2 = ones(nsamples,1);    
end

% Construct a vector to indicate when a genotype first shows up in a
% for-loop. This vector is used to determine when the genotype is shown in
% the figure.
novelgeno = [1; genoinx2(2:end)~=genoinx2(1:end-1)];

% Randomize color. Can be replaced by reset color group.
colormat = rand(size(genos_unique,1),3);

% Initiate the figure
figure('Color', [1 1 1])

hold on
for i = 1 : nsamples

    % Find out the current genotype index
    current_genoinx = genoinx2(i);
    
    for j = 1: nmaxbouts
        % Bout indices
        j1 = (j-1)*2+1;
        j2 = j*2;
        
        if ~isnan(data2(i,j1))
            % Draw the dash
            plot([data2(i,j1), data2(i,j2)], [i i], '-',...
                'LineWidth',6, 'Color', colormat(current_genoinx,:));
            
            % Compute duration
            duration(i,j) = data2(i,j2) - data2(i,j1);
        else
            % Otherwise no duration
            duration(i,j) = NaN;
        end
        
    end
    
    % Write out the genotypes
    if length(genos_unique) > 1 && novelgeno(i) == 1
        text2display = genos_unique{current_genoinx};
        text(265, i, text2display,...
            'Color', colormat(current_genoinx,:), 'FontSize', 11);
    end
end
hold off

% Draw a line at 2.5 hours (Optional)
% line([150,150],[0,size(data2,1)+0.5],'Color',[0.5 0.5 0.5],'LineWidth', 1)

% X-axis = 4.5 hours
xlim([0 270])

% Y-axis
ylim([0 size(data2,1)+0.5])

% Axis labels
ylabel('Fly number','FontSize',14)

%
% Set X ticks
% X-axis label
xlabel('Time in satiety assay (hours)','FontSize',14)
set(gca,'XTick',[0 60 120 180 240 300])
set(gca,'XTickLabel',[0 1 2 3 4 4.5])
%}

% Set font size
set(gca,'FontSize',14)
end
