% Functions for thresholding algorithms
% Last modified 28 April 2016 
function level = threshAlgo(img,type)
switch type
    case 1 % User selects Otsu.
        level = graythresh(img);
    case 2 % User selects Minimum.
        level = minimumthresh(img);
    case 3 % User selects Intermode.
        level = intermode(img);
    case 4 % User selects Isodata.
        level = isodata(img);
    case 5 % User selects Shangbag.
        level = shanbhag_mean(img);
end
end

function level = minimumthresh(I)
n=255;
y = imhist(I);
% Smooth the histogram by iterative three point mean filtering.
iter = 0;
while ~bimodtest(y)
    h = ones(1,3)/3; %low pass filter
    y = conv(y,h,'same'); %2D convolution
    iter = iter+1;
    % If the histogram turns out not to be bimodal, set T to zero.
    if iter > 10000;
        level = 0;
        warndlg('Histogram not bimodel after 10000 iterations')
        break
    end
end
% The threshold is the minimum between the two peaks.
for k = 2:n
    if y(k-1) > y(k) && y(k+1) > y(k)
        level = k;
    end
end
level = level/255;
end

function b = bimodtest(y)
% Test if a histogram is bimodal.
%  y    histogram
%  b    true if histogram is bimodal, false otherwise
len = length(y);
b = false;
modes = 0;
% Count the number of modes of the histogram in a loop. If the number
% exceeds 2, return with boolean return value false.
for k = 2:len-1
    if y(k-1) < y(k) && y(k+1) < y(k)
        modes = modes+1;
        if modes > 2
            return
        end
    end
end
% The number of modes could be less than two here
if modes == 2
    b = true;
    %     disp(['Number of modes = ',num2str(modes)]);
end
end


function level = intermode(I)
% J. M. S. Prewitt and M. L. Mendelsohn, "The analysis of cell images," in
% Annals of the New York Academy of Sciences, vol. 128, pp. 1035-1053, 1966.
% Assumes a bimodal histogram. The histogram needs is smoothed (using a
% running average of size 3, iteratively) until there are only two local maxima.
% Threshold t average gray levels of 2 peaks.
% Images with histograms having extremely unequal peaks or a broad and
% at valley are unsuitable for this method.

iter = 0;
level = -1;
iHisto = imhist(I);
tHisto = iHisto;

while ~bimodtest(iHisto)
    %smooth with a 3 point running mean filter
    for i=2:255
        tHisto(i)= (iHisto(i-1) + iHisto(i) + iHisto(i+1))/3;
    end
    tHisto(1) = (iHisto(1)+iHisto(1))/3; %0 outside
    tHisto(256) = (iHisto(255)+iHisto(256))/3; %0 outside
    iHisto = tHisto;
    iter=iter+1;
    if (iter>10000)
        level = -1;
        warndlg('Intermodes Threshold not found after 10000 iterations.')
        break
    end
end

%The threshold is the mean between the two peaks.
tt=0;
for i=2:255
    if (iHisto(i-1) < iHisto(i) && iHisto(i+1) < iHisto(i))
        tt = tt + i;
    end
end
level = floor(tt/2.0)/255;
end



function level=isodata(I)
% Iterative isodata method developed by T.W. Ridler, S. Calvard,
% Picture thresholding using an iterative selection method,
% IEEE Trans. System, Man and Cybernetics, SMC-8 (1978) 630-632.
% Step 1: Calculate mean gray values in the foreground and background
% (separated by initial guess of threshold (T(1) = mean of image)
% Step 2: New threshold value is computed as the average of 2 means
% Step 3: Repeat iteration until converge
% http://www.mathworks.com/matlabcentral/fileexchange/3195-automatic-thresholding/content/isodata.m
% Compute mean intensity from histogram (initial guess for thresh)
[counts,N]=imhist(I);
mu = cumsum(counts);
T(1) = round(sum(N.*counts)/mu(end));

% STEP 1: Compute background mean (MAT) and foreground mean (MBT)
mu2 = cumsum(counts(1:T(1)));
MBT = sum(N(1:T(1)).*counts(1:T(1)))/mu2(end);

mu3 = cumsum(counts(T(1):end));
MAT = sum(N(T(1):end).*counts(T(1):end))/mu3(end);

% Step 2: New threshold = (MAT+MBT)/2
T(2)=round((MAT+MBT)/2);

% STEP 3: Repeat iteration until converge
for i = 2:100000
    mu2 = cumsum(counts(1:T(i)));
    MBT = sum(N(1:T(i)).*counts(1:T(i)))/mu2(end);
    
    mu3 = cumsum(counts(T(i):end));
    MAT = sum(N(T(i):end).*counts(T(i):end))/mu3(end);
    T(i+1) = round((MAT+MBT)/2);
    
    if abs(T(i+1)-T(i)) < 1e-6 && abs(T(i)-T(i-1))<1e-6
        Threshold = T(i+1);
        break
    end
    
    if i==10000
        warndlg('Intermodes Threshold not found after 100000 iterations');
    end
end
% Normalize the threshold to the range [i, 1].
level = Threshold/255;
end

function level = shanbhag_mean(I)
% https://imagej.nih.gov/ij/source/ij/process/AutoThresholder.java
tot_ent = 0;  % total entropy
min_ent = 1e9;  % max entropy
ent_back = 0; % entropy of the background pixels at a given threshold
ent_obj = 0;  % entropy of the object pixels at a given threshold
norm_histo = zeros(1,256); %normalized histogram
P1 = zeros(1,256); %cumulative normalized histogram */
P2 = zeros(1,256);
total =0;
data = imhist(I);
for ih = 1:256
    total = total + data(ih);
end
for ih = 1:256
    norm_histo(ih) = data(ih)/total;
end
P1(1)=norm_histo(1);
P2(1)=1.0-P1(1);
for ih = 2:256
    P1(ih)= P1(ih-1) + norm_histo(ih);
    P2(ih)= 1.0 - P1(ih);
end
% Determine the first non-zero bin
first_bin=0;
for ih = 1:256
    if (abs(P1(ih))>2.220446049250313e-16)
        first_bin = ih;
        break;
    end
end
% Determine the last non-zero bin
last_bin=256;
for ih = 256:-1:first_bin
    if (abs(P2(ih))>2.220446049250313e-16)
        last_bin = ih;
        break;
    end
end

% Calculate total entropy each gray-level and find threshold that maximizes it
level =-1;
for it = first_bin:last_bin
    % Entropy of the background pixels */
    ent_back = 0.0;
    term = 0.5 / P1(it);
    for ih = 2:it
        ent_back = ent_back - norm_histo(ih) * log(1.0-term * P1(ih - 1));
    end
    ent_back = ent_back * term;
    
    % Entropy of the object pixels */
    ent_obj = 0.0;
    term = 0.5 / P2(it);
    for ih = it+1:256
        ent_obj = ent_obj - norm_histo(ih) * log(1.0-term * P2(ih));
    end
    ent_obj = ent_obj * term;
    
    % Total entropy
    tot_ent = abs(ent_back - ent_obj );
    
    if ( tot_ent < min_ent )
        min_ent = tot_ent;
        level = it;
    end
end
level = level/255;
end
