%%%%%
%
% Copyright (c) 2013, Rutgers Cancer Institute of New Jersey, 
%                     Center for Biomedical Imaging & Informatics
% All rights reserved.
%
% Redistribution and use in source and binary forms, with or without 
% modification, are permitted provided that the following conditions are 
% met:    
%    Redistributions of source code must retain the above copyright 
% notice, this list of conditions and the following disclaimer.
%    Redistributions in binary form must reproduce the above copyright notice, 
% this list of conditions and the following disclaimer in     
% the documentation and/or other materials provided with the distribution
%
% THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" 
% AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, 
% THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE 
% ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS 
% BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR 
% CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF 
% SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS 
% INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN 
% CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) 
% ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE 
% POSSIBILITY OF SUCH DAMAGE.
%
% License to use and modify this code is granted freely to all interested, 
% as long as the original authors arereferenced and attributed as such. 
% The original authors maintain the right to be solely associated with this work.
%
% Author: Wenjin Chen, Eugenia Xu
% Date: 2013/08/01
%%%%%
function [out, I2, I3] = activeContourCompute(fullfile, config)

display = 0;
printInit = 0;

coef = config.shrink;
conv = config.resolution;

seed = 30;
switch(config.size)
    case 1
        seed = 15;
    case 2
        seed = 20;
    case 3
        seed = 30;
    case 4
        seed = 45;
end

out= struct('folder', '', 'fn', '', 'goodResult', false, 'V', 0, 'PArea', 0, 'PMajorAxis', 0, 'PMinorAxis', 0, 'RArea', 0, 'RMajorAxis', 0, 'RMinorAxis', 0, 'adjfile', '', 'qa', '', 'Eccentricity', 0);

%disp(['computing ', fullfile]);

%% pre-processing

[I, folder, fn, ext, adjfile] = readImage(fullfile, config);

I2 = double(imresize(I, 1/coef));
II2 = double(I2);

h = size(I2, 1);
w = size(I2, 2);

%% compute location for initialization
mx = ones(h, 1)*mean(II2, 1);
my = mean(II2, 2)*ones(1, w);

Ix = mx-II2;
Ix(Ix<0)=0;
Iy = my-II2;
Iy(Iy<0)=0;

X = sum(Ix, 1);
Y = sum(Iy, 2);
 
[w2, h2, theseed] = generateSeed(X, Y, seed, coef);

%% construct the initial contour
bw = logical(zeros(h, w));
%s = ceil(size(seed, 1)/2);
s = theseed;
if s>=w2 
    w2=s+1; 
elseif (w-w2)<=s 
    w2 = w-s-1; 
end

if s>=h2 
    h2=s+1; 
elseif (h-h2)<=s 
    h2 = h-s-1; 
end
bw( (h2-s+1):(h2+s-1), (w2-s+1):(w2+s-1)) = getnhood(strel('disk', theseed, 8));


%% active contour
%[seg, phi, its] = chanvese(I2, bw, 500, .3, 2, 'r', 0);
%[seg, phi, its] = chanvese(I2, bw, 300, .5, 5, 'r', 0);

%[seg, phi, its] = chanvese(I2, bw, 500, .8, 10, 'r', 0); %shrink 10

[seg, phi, its] = chanvese(I2, bw, 300, .9, 5, 'r', 0);  %shrink is 5

if display
    disp(fullfile);
    disp([theseed its]);
end
seg = imopen(seg,  strel('disk',5));
a = regionprops(seg, {'Area', 'MajorAxisLength', 'MinorAxisLength', 'Eccentricity'});

 
%%

switch length(a)
    case 0   %nothing found, print all 0
        out.goodResult = false;
        out.PArea = 0;
        out.PMajorAxis = 0;
        out.PMinorAxis = 0;
        out.RArea = 0;
        out.RMajorAxis = 0;
        out.RMinorAxis = 0;
        out.Eccentricity = 0;
        out.V = 0;
        out.folder = folder;
        out.fn = [fn,  ext];
        out.adjfile = adjfile;
    case 1   %only one found, print
        out.goodResult = true;
        out.PArea       = a(1).Area*coef*coef;
        out.PMajorAxis  = a(1).MajorAxisLength*coef;
        out.PMinorAxis  = a(1).MinorAxisLength*coef;
        out.RArea       = a(1).Area*coef*coef*conv*conv;
        out.RMajorAxis  = a(1).MajorAxisLength*coef*conv;
        out.RMinorAxis  = a(1).MinorAxisLength*coef*conv;
        out.Eccentricity = a(1).Eccentricity;
        out.V   = .5e-9*a(1).MinorAxisLength*a(1).MinorAxisLength*a(1).MajorAxisLength()*((coef*conv)^3);
        out.folder = folder;
        out.fn = [fn,  ext];
        out.adjfile = adjfile;
    otherwise %more than one found, use largest
        a = regionprops(seg, {'Area', 'MajorAxisLength', 'MinorAxisLength', 'PixelIdxList', 'Eccentricity'});
        maxarea = 0;
        maxitem = [];
        for i = 1:size(a)
            x = a(i);
            if x.Area>maxarea
                maxarea = x.Area;
                maxitem=x;
            end
        end

        seg(:) = 0;
        seg(maxitem.PixelIdxList) = 1;

        out.goodResult = true;
        out.PArea       = maxitem.Area*coef*coef;
        out.PMajorAxis  = maxitem.MajorAxisLength*coef;
        out.PMinorAxis  = maxitem.MinorAxisLength*coef;
        out.RArea       = maxitem.Area*coef*coef*conv*conv;
        out.RMajorAxis  = maxitem.MajorAxisLength*coef*conv;
        out.RMinorAxis  = maxitem.MinorAxisLength*coef*conv;
        out.Eccentricity = maxitem.Eccentricity;
        out.V   = .5e-9*maxitem.MinorAxisLength*maxitem.MinorAxisLength*maxitem.MajorAxisLength()*((coef*conv)^3);
        out.folder = folder;
        out.fn = [fn, ext];
        out.adjfile = adjfile;
end
out.complete = 1;

%% output


seg = edge(seg);
I3 = uint8(zeros(h, w, 3));
I3(:, :, 1) = uint8(I2)+uint8(256*seg);
I3(:, :, 2) = uint8(I2)-uint8(256*seg);
I3(:, :, 3) = uint8(I2)-uint8(256*seg);

if printInit
    seg1 = edge(bw);
    I3(:, :, 1) = I3(:, :, 1)-uint8(256*seg1);
    I3(:, :, 2) = I3(:, :, 2)-uint8(256*seg1);
    I3(:, :, 3) = I3(:, :, 3)+uint8(256*seg1);
end

I3(h2, w2, 1) = uint8(0);
I3(h2, w2, 2) = uint8(255);
I3(h2, w2, 3) = uint8(255);

out.qa = [folder, filesep, fn, config.qc];
imwrite(I3, out.qa);



function [w2, h2, theseed] = generateSeed(X, Y, seed, coef)
X = X-min(X);
Y = Y-min(Y);

coeff = .4;
mx = max(X);
my = max(Y);

w2 = find(X==mx); w2 = w2(1);
h2 = find(Y==my); h2 = h2(1);

xlow = w2;
for i = w2:-1:1
    if X(i)<mx*coeff
        xlow = i;       break;
     end
end
xhigh = w2;
for i = w2:length(X)
    if X(i)<mx*coeff
        xhigh = i;       break;
    end
end
ylow = h2;
for i = h2:-1:1
    if Y(i)<my*coeff
        ylow = i;       break;
    end
end
yhigh = h2;
for i = h2:length(Y)
    if Y(i)<my*coeff
        yhigh = i;       break;
    end
end

theseed = seed;

if xlow~=w2 && xhigh ~=w2 && ylow~=h2 && yhigh~=h2
    theseed = ceil(min(xhigh-xlow, yhigh-ylow)/2)+ceil(40/coef);
    if theseed > max(length(X), length(Y))*.4
        disp('default seed');
        theseed = seed;
    end
else
    theseed = seed;
end