; NAME:
;   trajectory_maker
; PURPOSE:
;   Takes a list of track indices and matched trajectory indices from 
;   segregate_mts.pro, and the track array from track.pro, and reforms the track
;   array so that the last column is the trajectory each point is associated with,
;   not the track.  Should probably be part of segregate_mts.pro to ensure 
;   trajectories are actual points, not just lists of track numbers.
; CALLING SEQUENCE:
;   out = trajectory_maker(t,mt_t)
; INPUTS:
;   t:  7 by n element array of tracks generated by track.pro.  The zeroth
;       and first columns contain the x and y positions, the last column
;       contains the associated track number.  Each track is organized sequentially
;       in time.
;   mt_t:  A 2 by m element array, with track numbers in the 0th column, and 
;       trajectory number in the 1st column.  Each trajectory will be made up of
;       one or more tracks.
; OUTPUTS:
;   out:  A 7 by n element array (identical in size to t) with the first 6 columns
;       the same as in t, but the 6th (starting at 0) corresponding to the 
;       trajectory that particular point is associated with.
; SIDE EFFECTS:
;   None known.
; RESTRICTIONS:
;   Must run on t array with track number stored in the last (6th) column,
;   and with x and y locations in the 0th and 1st column.  
;   *** The trajectories produced are not ordered in time or space! ***  
;   They are just ordered by trajectory number.
; PROCEDURE:
;   For each trajectory, select each track associated with the trajectory, and
;   then for each track, get all points associated with the track, and identify
;   them (in the 6th column) with that trajectory.
;
; MODIFICATION HISTORY:
;   Written by DM, 10/10

; a simple function to combine the indexing of mt_t from segregate_mts.pro
; with the actual positions in t from track.pro.

function trajectory_maker,t,mt_t

  st=size(t) ; let's figure out how many elements are in t.
  out=fltarr(st(1)) ; a seed array...
  
; now, for each trajectory (stored in the 1st column) we'll go through and
; take each track and work with it...
  for i=0,max(mt_t(1,*)) do begin 
    tmp=fltarr(st(1)) ; temporary array to store this trajetory... 
    w=where(mt_t(1,*) eq i,ntracks) 
 
; now, go through and add each track to the output file.  Notice that this 
; dynamically resizes the output array.  Ok for our 10,000 element, 500 trajectory
; arrays.  Not ideal for 10^6 or something... 
    for j=0,ntracks-1 do begin ; now, just add each track..
      w2=where(t(6,*) eq mt_t(0,w(j))) ; the correct track...
      tmp=[[tmp],[t(*,w2)]] ; adding on this track ...
    endfor
    tmp=tmp(*,1:*) ; getting rid of the seed row.
    tmp(6,*)=i ; assigning it the trajectory #
    out=[[out],[tmp]] ; adding on to the output array.
  endfor
  
  return, out(*,1:*) ; again, getting rid of the seed row...

end