; NAME:
;   segregate_mts
; PURPOSE:
;   Takes a series of tracks (presumably of individual fluorophores on microtubules)
;   and works to separate them into individual trajectories.  Since many 
;   fluorophores are on each MT, many tracks overlap into a given trajectory.
;   However, separating tracks is complicated by crossing tracks, which are
;   put into a separate trajectory.
; CALLING SEQUENCE:
;   mt_t = segregate_mts(t,same_track=same_track)
; INPUTS:
;   t:  7 by n element array of tracks generated by track.pro.  The zeroth
;       and first columns contain the x and y positions, the last column
;       contains the associated track number.  Each track is organized sequentially
;       in time.
;   same_track:  a scalar keyword, which sets the distance the routine will search
;       for nearby tracks, in pixels.  Passed through to get_overlap.pro, where
;       the default is 3 pixels.
;   close_enough: a scalar keword sets the angle distance which nearby tracks are
;       accepted as the same...
; OUTPUTS:
;   mt_t(0,*):  Number corresponding to the track # in t(6,*).
;   mt_t(1,*):  The trajectory index each track is associated with.  Thus, this
;       variable ranges from 0 to ntrajectories.  Each trajectory is associated 
;       with one or more tracks in the 0th column.
; SIDE EFFECTS:
;   None known.
; RESTRICTIONS:
;   Must run on t array with track number stored in the last (6th) column,
;   and with x and y locations in the 0th and 1st column.  Beware parallel tracks
;   that are closer than same_distance apart - they will be the same.  Finally,
;   same_distance should be bigger than the track distance in track.pro.
; PROCEDURE:
;   Start with the first track (the 0th).  Find all other tracks that are within 
;   same_distance of every point in that track using get_overlap.pro.  Call those
;   tracks the same (as long as they are parallel).  Remove the original track from
;   the track array.  Make sure that all the other tracks are unique (many repeats
;   could be found since tracks overlap) and haven't been associated with a previous
;   trajectory.  Then, use the new tracks as seeds, using get_overlap.pro on
;   each in turn, and remove them from the track array so that no duplication is
;   found.  Once all the unique tracks associated with the original track are 
;   found, pick the next track, call it a new trajectory, and repeat.
;
; MODIFICATION HISTORY:
;   Written by DM, 10/10

; a simple function to segregate all microtubule trajectories in a track,
; t is from track.pro

; general idea: start with the first track, use get_overlap to find all overlapping
; parallel tracks, use those, add those in, reducing t, until finished.  Then,
; move on to next track and begin again...
function segregate_mts,t,same_track=same_track,close_enough=close_enough

  mt_t=fltarr(2,max(t(6,*))+1) ; an array to store the track numbers associated 
  ; with a given MT.  General ideal: [track#, mt#] so that each track gets attached
  ; to a given MT trajectory...
  t2=t ; this allows us to remove tracks while keeping the original array around.
  mt_idx=0
  out_idx=0
  
; now, as long as there are still tracks in the large track array (t2), keep going.  

  while t2(0) ne -1 do begin 
  
    idx=t2(6,0) ; the initial index (i.e. track number)
  
    over_tk=get_overlap(t2,idx,same_track=same_track,close_enough=close_enough) ; the initial overlapping tracks..
    w=where(t2(6,*) ne idx,nleft)
    if nleft eq 0 then begin ; I think we're at the last track..
      mt_t(0,out_idx)=idx
      mt_t(1,out_idx)= mt_idx
      t2=-1
    endif else begin
      t2=t2(*,w) ; the tracks reduced by index...
    endelse
    mt_t(0,out_idx)=idx
    mt_t(1,out_idx)= mt_idx
    out_idx=out_idx+1
 
; Ok, now use these tracks as seeds to find all the overlaping tracks of the
; secondary tracks (in over_tk), then these new secondary tracks become the seeds,
; until we can find no more overlapping tracks (i.e. over_tk = -1).
  
    while over_tk(0) ne -1 do begin ; if there were actually overlaping tracks..
; first, put the overlaping tracks into the output file with associated index.
      mt_t(0,out_idx:out_idx+size(over_tk,/n_ele)-1) = over_tk ; putting these new tracks
      mt_t(1,out_idx:out_idx+size(over_tk,/n_ele)-1) = mt_idx ; into the same MT idx.
      out_idx=out_idx+size(over_tk,/n_ele) ; incrementing the index in mt_t..

; now, using each of the secondary tracks as a seed, find all the tracks that overlap
; that track, and store in a new array called over_tk_tmp.

      over_tk_tmp=0 
      for i=0,size(over_tk,/n_ele)-1 do begin  ; I think I'm finding all the overlaps of these overlaps..
 
; grab all overlaping tracks into over_tk_t, just to make sure there were some.
  
        over_tk_t=get_overlap(t2,over_tk(i),same_track=same_track,close_enough=close_enough)
; if there are no tracks, then get_overlap returns -1.
        if over_tk_t(0) ne -1 then over_tk_tmp=[over_tk_tmp,over_tk_t]
       
; a quick check to see if we've run out of tracks - if so, end by setting t2=-1.
        w2=where(t2(6,*) ne over_tk(i),nother,compl=wc)
        if w2(0) ne -1 then  begin
          t2=t2(*,w2) ; not on the last track
        endif else begin
          t2=-1 ; on the last track...
        endelse
      endfor
      
; now, if there are tracks that overlap, make sure they are unique and then
; return to use them as the next seeds in the where loop.      
      
      if size(over_tk_tmp,/n_ele) eq 1 then begin
        ;print,'oops, no more tracks.
        over_tk=-1
      endif else begin
        over_tk=over_tk_tmp(1:*)
        s=sort(over_tk)
        over_tk=over_tk(s)
        u=uniq(over_tk)
        over_tk=over_tk(u)   ; the unique new tracks... now we can continue..
; except we need to eliminate those already counted        
        for j=0,size(over_tk,/n_ele)-1 do begin  
          w_over=where(mt_t(0,*) eq over_tk(j))
          if w_over ne -1 then over_tk(j)=-1 ; already in the list..
        endfor
        w_new=where(over_tk ne -1,nkeep)
        if nkeep ge 1 then over_tk=over_tk(w_new) ; these are really new tracks, not already counted...
      endelse
    endwhile
    mt_idx=mt_idx+1 ; next MT...
  endwhile
  
  
  return, mt_t

end