; NAME:
;   mt_tanangle
; PURPOSE:
;   Calculates the angle that the tangent to a trajectory makes with respect
;   to the x-axis.
; CALLING SEQUENCE:
;   ta = mt_tanangle(mtr)
; INPUTS:
;   mtr:  (nx,ny) array which contains the reduced x and y positions of
;          tracked MT data.  Must call mt_reduce.pro first.
; OUTPUTS:
;   ta(*): Contains the tangent angle at the position of the same index
;          in mtr.
; SIDE EFFECTS:
;   None, due to some useful temporary arrays.
; RESTRICTIONS:
;   Must operate on ordered trajectories.  Beware +/- pi shifts on extremely
;   noisy data.  Also, use only on trajectories that are approximately straight
;   on the 1 um scale.
; PROCEDURE:
;   For each point in the trajectory, do a linear fit to all data within 10 pixels.
;   The slope is the tangent to the trajectory, approximately.  However, using
;   ATAN (slope) results in a +/- pi problem.  Solve by looking at which way
;   the data are increasing in order (must be ordered!).  If the slope is
;   primarily vertical (i.e. abs(tan)>1) then look in the y coordinate to see
;   if increasing or decreasing.  If decreasing, at pi to the angle to get
;   the correct angle wrt. x-axis.  If primarily horizontal, (abs(tan)<1) do 
;   exactly the same by looking at the x coordinate.
;
; MODIFICATION HISTORY:
;   Written by DM, 9/10
;   

; a simple function to calculate the local tangent angle to the x axis
; for a bunch of microtubule points.
; just does a local linear fit, uses the slope, and picks the +/-pi 
; unknown by checking whether x is increasing or decreasing.  If increasing
; then ATAN(slope) gives the right angle.  If decreasing, then ATAN(slope) is off
; by pi.

function mt_tanangle,mtr,fitarea=fitarea
if not keyword_set(fitarea) then fitarea=10

sm=size(mtr)
ang_out=fltarr(sm(2))

for i=0,sm(2)-1 do begin ; for each point, get the local tangent angle!
; find the local MT region first...
  w=where(sqrt((mtr(0,*)-mtr(0,i))^2+(mtr(1,*)-mtr(1,i))^2) le fitarea, ngood)
  dx=mtr(0,w(ngood-1))-mtr(0,w(0)) ; check if nearly vertical...
  dy=mtr(1,w(ngood-1))-mtr(1,w(0))
  if abs(dy/dx) le 3 then begin ; more or less horizontal, good linear fit y(x)
    z=linfit(mtr(0,w),mtr(1,w)) ; do a linear fit.
  endif else begin
    z=linfit(mtr(1,w),mtr(0,w)) ; do a fit to x(y) instead...
    z(1)=1./z(1) ; and invert the slope
  endelse
  
  ang_out(i)=atan(z(1)) ; inverse tangent of the slope is the angle .. well, maybe.
  if abs(z(1)) ge 1 then begin ; now check vector direction, first for mostly vertical...
    dy=mtr(1,w(ngood-1))-mtr(1,w(0)) ; the difference in the y variable...
    if ((dy gt 0) and (ang_out(i) lt 0)) or ((dy lt 0) and (ang_out(i) gt 0)) $
       then ang_out(i)=ang_out(i)+!pi
  endif else begin ; then for mostly horizontal
    dx=mtr(0,w(ngood-1))-mtr(0,w(0)) ; the difference in the x variable...
    if dx lt 0 then ang_out(i)=ang_out(i)+!pi
  endelse
  
endfor

return, ang_out

end