; NAME:
;   mt_cosavg
; PURPOSE:
;   Takes the tangent angle as a function of position along a MT and
;   calculates the average of the cosine of the angle between points
;   as a function of distance between those points.  I.e. 
;   < cos (theta (ds)) > where ds is displacement between points.
;   Used for calculating persistence length (lp) of a track, which is defined
;   as <cos(theta(ds))> = exp(-ds/lp).  Also stores # of independent elements
;   in the average, so multiple measurements from different MTs can be combined.
; CALLING SEQUENCE:
;   ca = mt_cosavg(mtl,ta)
; INPUTS:
;   mtl:  n element array of positions along the MT at a given index.
;         Call mt_length.pro to generate.
;    ta:  n element array of tangent angles along the MT at the same
;         index as the positions in mtl above.  Call mt_tanangle.pro
; OUTPUTS:
;   ca(0,*):  Average displacements (ds) for the average cosines along the
;             MT.
;   ca(1,*):  Average of cosine of the angles for that displacement.
;   ca(2,*):  # of independent values, i.e. L/ds, where L is the length of the MT.
; SIDE EFFECTS:
;   None, due to some useful temporary arrays.
; RESTRICTIONS:
;   Must run on mtl and ta arrays that have corresponding indices.
; PROCEDURE:
;   First, calculate the displacement, ds, and the angle change, da, for
;   points separated by one index in mtl and ta using the shift routine.
;   Search through these for all points separated by ds ~ 1 pixel.  Then
;   repeat for 2, 3, etc.  Since there are gaps in positions along the
;   MT trajectory, sometimes a shift of 2 indices will result in a ds
;   of, say, 7.  Therefore, store all ds that are not equal to 2, and 
;   check in the next iteration, until all data are used.
;
; MODIFICATION HISTORY:
;   Written by DM, 9/10

; another "simple" (actually, complex) function to find the average
; of the cosines of the angles between two points (equivalent to the average
; dot product of the unit tangent vector) as a function of path length along the mt.
; the theory says that <cos(theta_ds)> = exp(ds/lp) where ds is the length along the 
; microtubule, and lp is the persistence length.  See PBoC, pp 290 or so.  Also, 
; Ott & Liebschaber, 1993, PRE on persistence length in actin.

; inputs: mtl, the pathlength along the microtubule as a function of position,
; in this case, index.  Use mt_length.pro
;         ta, the tangent angle at that index, from mt_tanangle.pro

function mt_cosavg, mtl,ta

sz=size(ta)
sm=size(mtl)
if sz(1) ne sm(1) then message,'Oops, not same size mt and tan angle'
tmpds=0 ; temporary variables for future use :)
tmpda=0
out=[0,1,mtl(sm(1)-1)] ; ds=0, cos(theta)=1, defined, so who knows how many meas...
for i=1,sz(1)-1 do begin
  ds=(mtl-shift(mtl,i))[i:sz(1)-1] ; ok, this complicated command does two things:
      ; first, it subtracts the current length from the length i over, i.e.
      ; the mtl-shift(mtl), then it only keeps the non-negative sides...
     
  da=(ta-shift(ta,i))[i:sz(1)-1]  ; now this calculates angle difference similarly
  tmpds=[tmpds,ds]
  tmpda=[tmpda,da] ; making a temporary storage array for our values... 
  ; now, where is our length the right length?
  w=where((tmpds ge i-0.5) and (tmpds lt i+0.5),ngood,complement=wc)
  if ngood ge 1 then begin
    out=[[out],[mean(tmpds(w)), mean(cos(tmpda(w))), mtl(sm(1)-1)/i]] 
    ; the output, with distance in first slot, mean cos(angle) in second and
    ; the number of independent measurements, L/ds in the third.
    tmpds=tmpds(wc)
    tmpda=tmpda(wc) ; let's remove the non-useful parts to speed up searching.
  endif else begin
    out=[[out],[0,0,0]] ; nothing...
  endelse
endfor

; ok, now we probably have some extra lengths hanging over us...
maxds=max(tmpds) ; the maximum displacement along the microtubule...
if maxds ge sz(1) then begin ; now, if there are more points to use..
  for i=sz(1),maxds do begin
    w=where(tmpds ge i-0.5 and tmpds lt i+0.5,ngood)
    if ngood ge 1 then begin
      out=[[out],[mean(tmpds(w)), mean(cos(tmpda(w))), mtl(sm(1)-1)/i]] ; the output, with distance in first slot, mean cos(angle) in second
    endif else begin
      out=[[out],[0,0,0]] ; nothing...
    endelse
  endfor
endif

return,out
end