; NAME:
;   mlsq
; PURPOSE:
;   Takes a MT trajectory and condenses it to its moving least
;   squares line (i.e. reduces spread) following Lee's 1999 paper.
; CALLING SEQUENCE:
;   out = mlsq(tk,range=range)
; INPUTS:
;   tk:  m by n element array of positions.  The 0th and 1st columns must contain
;       x and y respectively; the rest are just kept and returned.
;   range:  A scalar keyword which says how far away to look when doing the 
;       least squares fit.  Corresponds to H in Lee.  Default is 10 pixels.
; OUTPUTS:
;   out:  m by n element array (same size as tk), where the x and y positions have
;       been moved to their moving least squares positions.
; SIDE EFFECTS:
;   None known.
; RESTRICTIONS:
;   Must run on tk array with x and y in the 0th and 1st columns, respectively. 
;   Be aware:  If function is really curvy on the scale of range, then the
;   quadratic least squares fit will not be a good description of the data...
; PROCEDURE:
;   Takes data points scattered around a trajectory and thins them down to a 
;   trajectory that could be called a well-defined curve (not function).  
;   Follows
;   In-Kwon Lee, Computer Aided Geometric Design   Volume 17 ,  
;   Issue 2  (February 2000) Pages: 161 - 177  section 2
;   In essence, for each point, does a linear fit to all points weighted
;   by a Gaussian function (to localize) to find the slope.
;   Then, shifts a local region (+/- range) to the origin, and rotates to 
;   be approximately horizontal (using slope).  Then, does a quadratic least
;   squares fit, again with a weighting to count close points, ~ range/2, more,
;   falling to 0 at about range.  Shifts point to the fit point, then inverts
;   the rotation and shift to origin, so now we have our well-fit point.
;
; MODIFICATION HISTORY:
;   Written by DM, 10/10

; a simple function to implement the moving least squares algorithm of 
; In-Kwon Lee, Computer Aided Geometric Design   Volume 17 ,  
; Issue 2  (February 2000) Pages: 161 - 177  section 2
; in essence, to a linear fit, rotate local trajectory to the x-axis and
; shift to origin, do quadratic lsq fit, move to best fit, then transform back
; in essence, this is another version of mt_order, with the fitting aspect
; added in.
; So, let's first shift to the origin, then rotate...

function mlsq,tk,range=range

  if not keyword_set(range) then range=10 ; the distance to look over, corresponds to H
  st=size(tk)
  out=tk ; the output file

; now, let's operate on every point....

  for i=0,st(2)-1 do begin 
  
; the distance r, as described in Lee. 
    r=sqrt((tk(0,*)-tk(0,i))^2+(tk(1,*)-tk(1,i))^2) 
; weights (eq. 2 in Lee)    
    weights=exp(-r^2/range^2) 
; linear fit to find slope to make quadratic fit better.  Uses mpfitfun rather than
; LINFIT since LINFIT does not have a weights option.  Instead, has error, which 
; is the inverse of weights, so does not allow weighting of 0!    
    z=mpfitfun('linear',tk(0,*)-tk(0,i),tk(1,*)-tk(1,i),err,[0,1],weights=weights,/quiet)

; now for the rotation and quadratic fit part...
    w=where(r lt range,ngood)
    if ngood eq 0 then begin
      out(*,i)=tk(*,i) ; nothing nearby, just keep the point
    endif else begin
      tmptk=tk(0:1,w)

; shift to the origin..      
      tmptk(0,*)=tmptk(0,*)-tk(0,i)
      tmptk(1,*)=tmptk(1,*)-tk(1,i)
      
      ang=atan(z(1)) ; the angel wrt. x..
; rotation matrix, rotates through angle ang.
      rotmat=[[cos(ang), -sin(ang)],[sin(ang),cos(ang)]]
      
      rt=rotmat#tmptk ; rotating the temporary array parallel to x
; the weighting eq. 5 in Lee      
      qweights=2.0*r(w)^3/range^3 - 3*r(w)^2/range^2+1 
; quadratic least squares fit; once again mpfit fun to include weights.      
      qfit=mpfitfun('quadratic',rt(0,*),rt(1,*),err,[0,1,1],weights=qweights,/quiet)
; our fitted point rotated back..
      newpt=invert(rotmat)#[0.0,quadratic(0.0,qfit)] 
; and now we shift it back to the original position and store      
      out(0,i)=newpt(0)+tk(0,i)
      out(1,i)=newpt(1)+tk(1,i) 
    endelse
  endfor
  
  return,out

end