pro lugen_grid, x, xrange=xrange, nx=nx, $
                y, yrange=yrange, ny=ny, $
                z, zrange=zrange, nz=nz
                
;+
; The procedure lugen_grid generates 2D and 3D arrays of evenly
; spaced x, y, and z points.  Information about the third coordinate
; is optional and can be omitted, in which case the two-dimensional
; arrays will be produced.
;
; In the call to this procedure, x, y, z are arrays in which the 
; coordinates will be stored, xrange, yrange, zrange are two-component
; vectors giving the range of values on each axis, and nx, ny, nz are 
; the number of subdivision in each range.
;-

; Set nz appropriately if not supplied
if not keyword_set(nz) then nz=0

; Guard against entry of floating valus
nx=fix(nx) & ny=fix(ny) & nz=fix(nz)

; Set spacing between grid points; create arrays for x and y
dx = (xrange[1] - xrange[0])/float(nx)
dy = (yrange[1] - yrange[0])/float(ny)
x = fltarr(nx+1, ny+1, nz+1)
y = fltarr(nx+1, ny+1, nz+1)

; Generate lists of values of x and y
xtmp = xrange[0] + dx*findgen(nx+1)
ytmp = yrange[0] + dy*findgen(ny+1)

; Set spacing and create array for z if necessary
if nz ne 0 then begin
  dz = (zrange[1] - zrange[0])/float(nz)
  z = fltarr(nx+1, ny+1, nz+1)
  ztmp = zrange[0] + dz*findgen(nz+1)
endif

; Write valus into arrays for x, y, and (if necessary) z
for i = 0, nx do begin
  for j = 0, ny do begin
    for k = 0, nz do begin
      x[i,j,k] = xtmp[i]
      y[i, j, k] = ytmp[j]
      if nz ne 0 then z[i,j,k] = ztmp[k]
    endfor
  endfor
endfor

end