; NAME:
;   local_angle
; PURPOSE:
;   Finds the tangent angle to a trajectory at a given point.
; CALLING SEQUENCE:
;   ang_out = local_angle(traj,pos,fitarea=fitarea)
; INPUTS:
;   traj:  The trajectory we want to find the tangent angle to.
;   pos:   The point at which we want to find the tangent
;   fitarea:  a scalar keyword which sets the distance the routine will search
;       for neighboring points in order to do the fit.  Default is 10 pixels.
; OUTPUTS:
;   ang_out:  Tangent angle with respect to the x-axis.
; SIDE EFFECTS:
;   None known.
; RESTRICTIONS:
;   Needs at least one other point within fitarea in order to do a linear fit.
;   Be ware: if fitarea is longer than the trajectory is approximately straight,
;   this will not find the tangent angle, since it just uses a linear fit.
; PROCEDURE:
;   Finds every point within fitarea of pos.  Then, checks to see if the line is
;   more or less horizontal or vertical - if vertical, invert x and y to get a 
;   better slope out.  Fit either x,y or y,x, take slope, and then use the first
;   and last point to figure out which direction the tangent vector should point -
;   i.e. resolves the pi degeneracy of the slope.  Return that angle wrt. the x-axis.
; MODIFICATION HISTORY:
;   Written by DM, 10/10

function local_angle,traj,pos,fitarea=fitarea

  if not keyword_set(fitarea) then fitarea=10

; find all nearby points.
  w=where(sqrt((traj(0,*)-pos(0))^2+(traj(1,*)-pos(1))^2) le fitarea, ngood)
  
; check if vertical or horizontal using the dy and dx displacements.  
  dx=traj(0,w(ngood-1))-traj(0,w(0)) 
  dy=traj(1,w(ngood-1))-traj(1,w(0))
  if abs(dy/dx) le 3 then begin ; more or less horizontal, good linear fit y(x)
    z=linfit(traj(0,w),traj(1,w)) ; do a linear fit.
  endif else begin
    z=linfit(traj(1,w),traj(0,w)) ; do a fit to x(y) instead...
    z(1)=1./z(1) ; and invert the slope
  endelse
  
  ang_out=atan(z(1)) ; inverse tangent of the slope is the angle .. well, maybe.
  
; now check vector direction.  If mostly vertical (first), then see which 
; quadrant the line falls in.  If 4 and dy > 0, then send around to 2 (i.e. up).
; if 1 and dy < 0, then send around to 3 (down), etc.
  if abs(z(1)) ge 1 then begin 
    if ((dy gt 0) and (ang_out lt 0)) or ((dy lt 0) and (ang_out gt 0)) $
       then ang_out=ang_out+!pi
  endif else begin ; then for mostly horizontal
    if dx lt 0 then ang_out=ang_out+!pi
  endelse
  
  return,ang_out

end