; NAME:
;   get_overlap
; PURPOSE:
;   Takes one track as identified using track.pro, and then finds all other
;   tracks that "overlap" this track, in the sense that they are essentially
;   parallel (to within close_enough, 0.1 radians), and nearby (to within
;   same_track pixels, default of 3.0).  Returns the indices of the overlaping
;   tracks.
; CALLING SEQUENCE:
;   out = get_overlap(t,idx,same_track=same_track)
; INPUTS:
;   t:  7 by n element array of tracks generated by track.pro.  The zeroth
;       and first columns contain the x and y positions, the last column
;       contains the associated track number.  Each track is organized sequentially
;       in time.
;   idx:  The number (index) of the track to find overlaps for.
;   same_track:  a scalar keyword, which sets the distance the routine will search
;       for nearby tracks, in pixels.  Default is 3 pixels.
;   close_enough:  a scalar keword sets the angle distance which nearby tracks are
;       accepted as the same...
; OUTPUTS:
;   out:  Ordered list of the tracks (indices of the tracks) which overlap the
;       input track.  Each overlaping track is listed once.
; SIDE EFFECTS:
;   None known.
; RESTRICTIONS:
;   Must run on t array with track number stored in the last (6th) column,
;   and with x and y locations in the 0th and 1st column.  Beware parallel tracks
;   that are closer than same_distance apart - they will be the same.  Finally,
;   same_distance should be bigger than the track distance in track.pro.
; PROCEDURE:
;   Starts with the first point of the track.  Looks for all other points on
;   different tracks which are within same_distance of the original point.  
;   Picks the closest point from each nearby track and checks whether it 
;   is parallel using local_angle.pro to find the local_angle of the other
;   track.  If the angles are closer than close_enough, call it an overlaping
;   track.  If not, don't.  Repeat for every point in the original track.
;   Finally, make sure each overlap is only listed once in out, and return.
; MODIFICATION HISTORY:
;   Written by DM, 10/10

function get_overlap,t,idx,same_track=same_track,close_enough=close_enough

; how close do they need to be to be considered the same track?
  if not keyword_set(same_track) then same_track=3.0
  
; how close in angle do they need to be to be considered teh same track?
  if not keyword_set(close_enough) then close_enough=0.1 ; the level of parallelness of adjacent tracks to be the same...

; find the track itself, and all other tracks (wc).  Be very careful, since
; wc now is the container for all future indices...  
  w=where(t(6,*) eq idx,ngood,complement=wc) ; the track itself.

  if wc(0) eq -1 then begin
    out=-1 ; no overlaps because this is the only track
  endif else begin
  
    tmpt=t(*,w) ; the track we're interested in..
    parallel=0 ; the parallel trajectories
    non_par=0 ; the non-parallel trajectories
    ta=mt_tanangle(tmpt) ; the tangent angle on this segment of track...

; now, for every point in the track, find the overlaping tracks in parallel,
; and the crossing tracks in nonpar.

    for i=0,ngood-1 do begin
    
; looking for other nearby tracks (i.e. < same_track pix away in a different 
; track, hence the complement wc
    
      w2=where(sqrt((t(0,wc)-tmpt(0,i))^2+(t(1,wc)-tmpt(1,i))^2) le same_track,ng) 

; now to check if those other tracks are parallel or not..
      if ng ge 1 then begin 
        u=uniq(t(6,wc(w2))) ; the unique nearby tracks... - only within the complement, wc
        for j=0,size(u,/n_ele)-1 do begin ; for each track
          w3=where(t(6,*) eq t(6,wc(w2(u(j))))) ; the indices of the track..
          ; find the closest point to our point...
          mindist=min(sqrt((t(0,w3)-tmpt(0,i))^2+(t(1,w3)-tmpt(1,i))^2),closest)
; use local_angle to get the local angle of that track at the closest point.
          la=local_angle(t(*,w3),t(*,w3(closest)))
          
; check if parallel.  Remember, for parallel, the angle needs to be 0 or pi, since
; the direction is just an accident of the where search...          
          if (abs(la-ta(i)) le close_enough) or $
             (abs(abs(la-ta(i))-!pi) le close_enough) then begin
            parallel=[parallel,t(6,wc(w2(u(j))))]
          endif else begin
            non_par=[non_par,t(6,wc(w2(u(j))))]
          endelse
        endfor
      endif
    endfor
    
    if size(parallel,/n_ele) eq 1 then begin ; oops, no overlaps found
      out=-1
    endif else begin
; keep the unique outputs.    
      par=parallel(1:*)
      s=sort(par)
      sp=par(s)
      u_out=uniq(sp)
      out=sp(u_out)
    endelse
  endelse
  return,out

end
