; NAME:
;   get_lp
; PURPOSE:
;   A script to calculate the persistence length of microtubules from a MT gliding assay video (tiff stack).
; CALLING SEQUENCE:
;   @get_lp.pro
; INPUTS:
;   Nothing directly.  Will open dialog box to pick the tiff stack to act on.
; OUTPUTS:
;   A script, so no return.  However, the following variables will be created:
;   image:  3-dimensional array storing the underlying image data.
;   pt:     array storing the positions of every fluorophore in every frame
;   t:      array of trajectories of individual fluorophores
;   mt_t:   list of trajectory number and associated microtubule trajectory number.
;   mtt:    array of individual fluorophore trajectories with associated microtubule trajectories.
;   mlmt:   thinned array of microtubule trajectories from single fluorophore trajectories
;   omt:    array of microtubule trajectories ordered from one end.
;   lomt:   ordered array of microtubule trajectories 50 pixels or longer
;   ca:     average cosine data from all trajectories in the video
;   lp:     the persistence length, in pixels
; SIDE EFFECTS:
;   Will overwrite variables with the names above.
; RESTRICTIONS:
;   Must run on greyscale tiff stacks.  Relies on a hard-coded intensity threshold
;   for individual fluorophore intensity, MIN_INTENSITY.  We've found that values of
;   5000-10000 work for pixels that are 100 nm, with peak intensities of about 1000 counts.
;   However, this value will need to be changed depending on camera, laser, and microscope settings.
; PROCEDURE:
;   1. Reads in given tiff stack.
;   2. Tracks each flurophore, compiles trajectories of individual fluorophores.
;   3. Combines trajectories of all fluorophores on a microtubule into a composite microtubule trajectory.
;   4. Thins composite trajectory into a single trajectory, one point per pixel (or so).
;   5. Calculates <cos(theta_s)>, where <> denotes average, for the relative tangent angle theta between
;      positions a distance s apart along the trajectory.
;   6. Fits <cos(theta_s)>=exp[-s/lp] to find the persistence length, lp
;   7. Plots fits, and writes plots to postcript files with same base filename as the original
;      tiff stack.  Also writes cosine_average data as .gdf file.
;
; MODIFICATION HISTORY:
;   Written by DM, 10/10

DISTANCE=5.0 ; the distance over which to look for the same track..
MIN_INTENSITY=5000
delvar,image
; if there was a previous image, delete it.  Otherwise, we run out of memory..
imagefile=dialog_pickfile()
image = readtiffstackfm(imagefile) ; our file...
pt=1
dpretrack,image,mass=MIN_INTENSITY,outvar=pt,/quiet ; creates the array (pt) of all fluorophore positions
; careful with the masscut - 5000 is good for 5 Hz, 3 mW illumination at
; a gain of 3000 on the ANDOR iXon..
t=track(pt,3.0,goodenough=5,memory=3,/quiet)
; combines individual fluorophore positions into fluorophore trajectories, see
; http://www.physics.emory.edu/~weeks/idl/tracking.html
; for an overview.

mt_t=segregate_mts(t,same_track=DISTANCE)

; this links up all tracks within 8.0 pixels of eachother into the same track
; note: same_track had better be bigger than the second number in track!
; note: this also ignores crossing tracks (or, at least, tracks that differ by
; more than 0.1 radian)  However, mt_t is just a list of track number and mt number.

mtt=trajectory_maker(t,mt_t) 

; this actually takes the list that is mt_t and turns it into a series
; of trajectories...

mlmt=mt_mlsq(mtt)

; this shrinks all trajectories to their moving least squares positions - 
; essentially reducing noise along the trajectory, with an up to 10 pixel
; distance (although weighting falls off steeply beyond 5 pixels).  
; This will affect the very shortest distances in lp - so we need to ignore
; numbers up to 5 pixels!!

omt=order_mt(mlmt,nearest=DISTANCE) ; finds ends and orders the MTs..

lomt=longer_trajectories(omt,50)  ; only long trajectories (50 pixels) are worth keeping
ca=all_ca(lomt,/i,nearest=DISTANCE)
; above finds the cosine average for all of the tangent angles.

w=where(ca(2,*) ne 0) ; as long as there are actual data points...

lp=mpfitfun('nexp',ca(0,w),ca(1,w),err,[1000.0],/quiet,weights=sqrt(ca(2,w)))
; fits the data to 'nexp' - <cos(theta_s)>=exp(-s/lp)  
; one word of caution: the weighting is not perfect.  Large s are overweighted, due
; to the technique for calculating <cos(theta_s)> - overlapping regions are used.

ntraj=max(lomt(6,*))+1 ; number of trajectories...
window,0
; the remainder simply plots and saves the data.

plot, lomt(0,*),lomt(1,*),psym=3,/iso,/yno,xtitle='pixels',ytitle='pixels', $
  title='all '+strcompress(string(uint(ntraj)))+' trajectories from '+imagefile, $
  xrange=[0,512],yrange=[0,512]
window,1
plot,ca(0,*),ca(1,*),psym=3,xtitle='trajectory length (pix)',ytitle='<cos(theta)>', $
  title='average cosines from '+imagefile
oplot,ca(0,w),nexp(ca(0,w),lp),color=255
xyouts,10,0.5,'persistence length = '+strcompress(string(lp))+' pix'
set_plot,'ps'
device, filename=strmid(imagefile,0,strlen(imagefile)-4)+' trajectories.ps'
plot, lomt(0,*),lomt(1,*),psym=3,/iso,/yno,xtitle='pixels',ytitle='pixels', $
  title='all '+strcompress(string(uint(ntraj)))+' trajectories from '+imagefile, $
  xrange=[0,512],yrange=[0,512]
device,/close
device, filename=strmid(imagefile,0,strlen(imagefile)-4)+' cosavg.ps'
plot,ca(0,*),ca(1,*),psym=3,xtitle='trajectory length (pix)',ytitle='<cos(theta)>', $
  title='average cosines from '+imagefile
oplot,ca(0,w),nexp(ca(0,w),lp)
xyouts,10,0.5,'persistence length = '+strcompress(string(lp))+' pix'
device,/close
set_plot,'win'
write_gdf,ca,strmid(imagefile,0,strlen(imagefile)-4)+' cosavg.gdf'

