; NAME:
;   get_ca
; PURPOSE:
;   Takes a trajectory that has already been thinned by mlsq.pro and finds
;   the <cos(theta)> as a function of length along the MT, as well as the
;   implicit number of independent values that are in the average.  Used for
;   finding persistence length lp in <cos(theta(ds))> = exp -ds/lp.
; CALLING SEQUENCE:
;   ca = get_ca(tk,taout=taout)
; INPUTS:
;   tk:  A trajectory of x, y positions (in 0th and 1st columns), pre thinned by mlsq.
;   taout:  set to a named variable which will contain the tangent angles along
;       the trajectory for error checking and diagnostics.
; OUTPUTS:
;   ca:  The <cosines> (i.e. cosines averaged).  0th column contains the pathlength
;       over which the cosine was computed (i.e. ds).  1st column contains the
;       <cos> value.  2nd column contains number of independent values for weighting
;       and combining tracks.
; SIDE EFFECTS:
;   None known.
; RESTRICTIONS:
;   Must use with tk array with x in 0th column, y in 1st column. 
; PROCEDURE:
;   First, finds the end of the trajectory using endfinder.pro.  Then, orders
;   the positions of the trajectory as a function of pathlength from that end
;   using mt_order.pro. Then, reduces the density of points to about 1 per pixel, 
;   using mt_reduce.pro.  Gets the pathlength from the end using mt_length.pro, and
;   the tangent angle to each point using mt_tanangle.pro.  Finally, uses
;   mt_cosavg.pro to get the cosine averages.  See each file for how it works.
;
; MODIFICATION HISTORY:
;   Written by DM, 10/10

; a simple function to get the combined cosine averages for a track.
; needs to follow the mt_mlsq.pro routine.  Works on a single track.
function get_ca,tk,taout=taout,nearest=nearest

  mtend=endfinder(tk)
  mto=mt_order(tk,firstpoint=mtend,nearest=nearest)
  mtr=mt_reduce(mto)
  if (size(mtr))[0] eq 1 then begin ; oops, a 1 pixel trajectory...
    ca = -1
  endif else begin
    mtl=mt_length(mtr)
    ta=mt_tanangle(mtr,fitarea=15)
    ;dta=(ta-shift(ta,1))[1:*] ; the change in tangent angle...
    ;w=where(abs(dta) lt 0.1,ncomp=ncomp) ; so, let's exclude giant jumps...
    ;ca=mt_cosavg(mtl([0,w+1]),ta([0,w+1]))
    ca=mt_cosavg(mtl,ta)
    taout=ta ; just allows us to return ta for error analysis...

  endelse
  return,ca
end