;+
;
;
; NAME:
;   gaussfit_feature 
; PURPOSE:
;   Does a Gaussian fit to all particles found in a feature call, 
;   returning centers, widths, and peaks.
; CATEGORY:
;   Image Processing
; CALLING SEQUENCE:
;   gf = gaussfit_feature(image, f, mask_size)
; INPUTS:
;   image:  (nx,ny) array which presumably contains some
;     features worth fitting
;   f:  output from a feature call on the same image
;   mask_size:  Size of the region to fit.
; OUTPUTS:
;   raw[0,*] is x0, the x-coordinate of the center
;   raw[1,*] is y0, the y-coordinate of the center
;   raw[2,*] is sigX, one standard-deviation in x
;   raw[3,*] is sigY, one standard-deviation in y
;   raw[4,*] is A, the amplitude scaling factor
;   raw[5,*] is the constant background term
; SIDE EFFECTS:
;   Hopefully none.
; RESTRICTIONS:
;   To work properly, the features should be separated
;   by mask_size or greater numbers of pixels - otherwise, 
;   fits will be off.
; PROCEDURE:
;   Extracts an area around each feature (f(*,i) as initial guess parameters), 
;   calls gaussfit2d on all images, using f(*,i).  Adds back in original
;   positions, and returns.
; 
;-
;  Goes with dpretrack - does a gaussian fit to all the particles from feature.
;  Based on dave meichle's gofit.pro




FUNCTION gaussfit_feature, image, f, mask_size

!QUIET=1
!EXCEPT=0
; blinking check - if the spot is too big, ignore...
MAX_WIDTH=mask_size 
;generate indices to address extracted area.

lugen_grid, x, xrange=[0,mask_size], nx = mask_size, $
y, yrange = [0, mask_size], ny = mask_size

NUM_OF_FEATURES = n_elements(f(0,*))
; a little error checking...
if f(0) eq -1 then message,'No elements in feature.  Stopping gaussfit_feature()....'

sz = size(image)
if sz(0) ne 2 then message, 'Image is not a 2D array.  Use a 2D array'

;determine the chunk to extract from the image
;  first time use position from f(), and don't use non-image parts :)
xmin = round(f(0,*) - mask_size/2.) 
ymin = round(f(1,*) - mask_size/2.) 
zero = where(xmin lt 0, count)
if(count gt 0) then xmin(zero) = 0
zero = where(ymin lt 0, count)
if(count gt 0) then ymin(zero) = 0
zero = where(xmin+mask_size ge sz(1), count)
if(count gt 0) then xmin(zero) = sz(1)- mask_size - 1
zero = where(ymin+mask_size ge sz(2), count)
if(count gt 0) then ymin(zero) = sz(2) - mask_size - 1


;chunk = fltarr(mask_size+1, mask_size+1)

p_in = fltarr(6,NUM_OF_FEATURES)


; Modifying to take into account blinking...
p_out = fltarr(6, NUM_OF_FEATURES)


for k = 0, (NUM_OF_FEATURES-1) do begin 
    
; extract chunk from image to work with curve-fitting
  chunk = image(xmin(k):(xmin(k)+mask_size), ymin(k):(ymin(k)+mask_size)) 
; initialize INITIAL  p_in(*,k) with parameters from f[]
  p_in = [f(0,k)-xmin(k),f(1,k)-ymin(k),sqrt(f(3,k))/2.,sqrt(f(3,k))/2.,max(chunk),mean(chunk)] 
; call fitting routine on the chunk
        ;p_out(*,j,k) = mpfit2dfun('myGauss',x,y,chunk,1,p_in,/quiet)
  ;outtmp=mpfit2dfun('myGauss',x,y,chunk,1,p_in,/quiet) ;(slow)
  yfit=gauss2dfit(chunk,out2)
  outtmp=[out2(4),out2(5),out2(2),out2(3),out2(1),out2(0)]
  ; Now, test whether this is a "good" point:
  if  $  ;     First, is x in range?
    outtmp(0) gt mask_size or outtmp(0) lt 0 or $ ; next is y in range?
    outtmp(1) gt mask_size or outtmp(1) lt 0 or $ ; next widths - too large
    outtmp(2) gt MAX_WIDTH or outtmp(3) gt MAX_WIDTH or $ ; next, amplitude
    outtmp(4) lt 0 then begin
;   do nothing!
  endif else begin
    p_out(*,k)=outtmp
     
    ;add xmin and ymin offset to x and y locations
    p_out(0,k) = p_out(0,k) + xmin(k) 
    p_out(1,k) = p_out(1,k) + ymin(k)
      
  endelse 
endfor 

!QUIET=0
RETURN, p_out

END