; NAME:
;   ca_combiner
; PURPOSE:
;   Takes the <cos> from two tracks and combines them into one <cos> as a function
;   of ds, the distance along the trajectory.
; CALLING SEQUENCE:
;   out = ca_combiner(ca1,ca2)
; INPUTS:
;   ca1:  An array of <cos(theta(ds))> as a function of ds.  Col 0: ds.  
;       Col 1: <cos>.  Col 2: Number of independent measurements, i.e. L/ds, where
;       L is the path length of the trajectory (microtubule).
;   ca2:  A second array of <cos(theta(ds))>
; OUTPUTS:
;   out:  An array of <cos(theta(ds))> as above, with both original arrays combined
;       into the output array.
; SIDE EFFECTS:
;   None known.
; RESTRICTIONS:
;   None, I think.
; PROCEDURE:
;   First, picks out the shorter and longer (in terms of L) trajectory.  The longer
;   values are used for ds> the shorter L.  For short ds, the new <cos> is calculated
;   by averaging the <cos> from the other - i.e. 
;   <cos>_out = (<cos>1*N1+<cos>2*N2)/(N1+N2), where N1 and N2 are the number of
;   independent measurements in <cos>, so that the average is the same as if the 
;   underlying data were averaged.
;
; MODIFICATION HISTORY:
;   Written by DM, 10/10

; a simple way to combine two <cos(theta)> vs. ds files...

function ca_combiner,ca1,ca2

s1=size(ca1)
s2=size(ca2)

longest=max([s1(2), s2(2)],whichone) ; the longest length

out=fltarr(3,longest)

if whichone eq 0 then begin ; the first was longer...
  s=s2
  shorter=ca2
  longer=ca1
endif else begin ; the second was longer
  s=s1
  shorter=ca1
  longer=ca2
endelse
if shorter(0) eq -1 then begin ;oops, one of the CA's failed on the way in..
  out = longer
endif else begin
; need to check on the unlikely case that ca1 and ca2 are the same length!!!
  if size(out(0,*),/n_ele) ne s(2) then $ ; not the same length...
    out(*,s(2):*)=longer(*,s(2):*) ; put all of the second afterward..
    
  out(0,0:s(2)-1)=(longer(0,0:s(2)-1)*longer(2,0:s(2)-1)+shorter(0,*)*shorter(2,*)) / $
        (longer(2,0:s(2)-1)+shorter(2,*)) ; the ds values...
  out(1,0:s(2)-1)=(longer(1,0:s(2)-1)*longer(2,0:s(2)-1)+shorter(1,*)*shorter(2,*)) / $
        (longer(2,0:s(2)-1)+shorter(2,*)) ; the <cos> values
  out(2,0:s(2)-1)=longer(2,0:s(2)-1)+shorter(2,*) ; the new number of indep. values.
endelse
return, out

end
