; NAME:
;   all_ca
; PURPOSE:
;   Takes trajectories that have already been thinned by mt_mlsq.pro and finds
;   the <cos(theta)> as a function of length along the trajectories, as well as the
;   implicit number of independent values that are in the average.  Used for
;   finding persistence length lp in <cos(theta(ds))> = exp -ds/lp.
; CALLING SEQUENCE:
;   caout = all_ca(mlmt)
; INPUTS:
;   mlmt:  A 7 by n list of trajectories, thinned by mt_mlst (m), and shortened
;       by long_trajectories.pro (l).  First two columns are x and y, last is
;       trajectory number.
;  ignore_jumps:  A keyword set to ensure that trajectories with big jumps
;       in the tangent angle (corresponding to accidentally connected crossing
;       trajectories) are ignored.
; OUTPUTS:
;   caout:  The <cosines> (i.e. cosines averaged).  0th column contains the pathlength
;       over which the cosine was computed (i.e. ds).  1st column contains the
;       <cos> value.  2nd column contains number of independent values for weighting
;       and combining tracks.
; SIDE EFFECTS:
;   None known.
; RESTRICTIONS:
;   Must use with mlmt array with x in 0th column, y in 1st column, traj. # in last.
; PROCEDURE:
;   For each trajectory, calls get_ca.pro.  Then, combines the <cos> using 
;   ca_combiner.pro to get all of them from a given experiment.  Spits out warnings
;   if there are tracks too short ("bad track") to use, or have big changes in 
;   tangent angle ("big jump").  Excludes bad tracks, but includes big jumps.
;   Should look at tracks with big jumps to ensure they're not actually bad.
;
; MODIFICATION HISTORY:
;   Written by DM, 10/10

; a simple function get get and combine all the cosine averages from a single
; trajectory list, already lined up using mt_mlsq.pro

function all_ca,mlmt,ignore_jumps=ignore_jumps,nearest=nearest


  for i=0,max(mlmt(6,*)) do begin ; for each trajectory...
    ta=1
    w=where(mlmt(6,*) eq i)
    tk=mlmt(*,w)
    ca=get_ca(tk,taout=ta)
    if ca(0) eq -1 then begin
      print,'danger, bad track # '+strcompress(string(i))
    endif else begin
      tatmp=(ta+2*!pi) mod (2*!pi) ; to handle 360 degree changes...
      dta=(tatmp-shift(tatmp,1))[1:*] ; the change in tangent angle...
      w=where((abs(dta) gt 0.2) and (abs(dta) lt 5),nbig) ; so, let's exclude giant jumps...
      if nbig ge 1 then begin
        print,'danger, big jump...'+strcompress(string(i))
        if keyword_set(ignore_jumps) then  ca=-1 ; most of these consist of two tracks which were accidentally joined..
      endif
    endelse
    if i ge 1 then begin
      caout=ca_combiner(ca,caout)
    endif else begin
      caout=ca ; on the first round, store the current ca in the output ca..
    endelse
  endfor
  
  return,caout

end
